# 🔒 Güvenlik Dokümantasyonu

## Dans Dersleri - Güvenlik Önlemleri

### 🛡️ Uygulanan Güvenlik Önlemleri

#### 1. **Authentication (Kimlik Doğrulama)**

##### Login Sistemi
- ✅ E-posta ve şifre bazlı giriş
- ✅ Şifre karmaşıklık gereksinimleri:
  - Minimum 8 karakter
  - En az 1 büyük harf
  - En az 1 küçük harf
  - En az 1 rakam
  - En az 1 özel karakter (!@#$%^&*)

##### Demo Hesaplar (Sadece Development Ortamı İçin)
```
ÖNEMLİ: Production'da bu hesaplar KESİNLİKLE olmamalıdır!
Development ortamında test için kullanılan demo hesaplar ayrı bir
environment variable dosyasında tutulmalı ve production'a deploy
edilmemelidir.
```

#### 2. **Rate Limiting (Hız Sınırlama)**
- ✅ 5 başarısız giriş denemesi sonrası 15 dakika kilitleme
- ✅ IP bazlı takip
- ✅ Kalan deneme hakkı gösterimi

#### 3. **Session Management (Oturum Yönetimi)**
- ✅ Token tabanlı session sistemi
- ✅ 24 saat session süresi
- ✅ Auto-logout on session expiry
- ✅ "Beni Hatırla" özelliği

#### 4. **Authorization (Yetkilendirme)**

##### Rol Sistemi
- **Super Admin**: Tüm yetkiler
- **Admin**: Kullanıcı ve kurs yönetimi
- **Editor**: Kurs oluşturma ve düzenleme
- **Moderator**: İçerik moderasyonu

##### Yetki Kontrolleri
```typescript
// Örnek kullanım
hasPermission(user.role, 'courses.delete') // true/false
```

#### 5. **Route Protection (Rota Koruması)**
- ✅ Middleware ile otomatik yönlendirme
- ✅ Protected admin routes
- ✅ Session kontrolü her istekte

#### 6. **XSS (Cross-Site Scripting) Koruması**
- ✅ Input sanitization
- ✅ Output encoding
- ✅ Content Security Policy headers

```typescript
// Input temizleme
sanitizeInput(userInput)
```

#### 7. **SQL Injection Koruması**
- ✅ Parametreli sorgular
- ✅ Input escape
- ✅ ORM kullanımı (production'da)

```typescript
// SQL escape
escapeSQL(value)
```

#### 8. **CSRF (Cross-Site Request Forgery) Koruması**
- ✅ CSRF token üretimi
- ✅ Token doğrulama
- ✅ SameSite cookie attribute

```typescript
// Token oluşturma
const csrfToken = generateCSRFToken()
```

#### 9. **Audit Logging (Denetim Kaydı)**
- ✅ Tüm admin işlemleri loglanır
- ✅ IP adresi ve user agent kaydı
- ✅ Timestamp tracking

```typescript
createAuditLog({
  userId: user.id,
  action: 'login',
  resource: 'admin_panel',
  timestamp: new Date().toISOString(),
  ip: req.ip,
  userAgent: req.headers['user-agent']
})
```

### 🔐 Production İçin Ek Öneriler

#### 1. **HTTPS Zorunluluğu**
```javascript
// next.config.js
module.exports = {
  async headers() {
    return [
      {
        source: '/:path*',
        headers: [
          {
            key: 'Strict-Transport-Security',
            value: 'max-age=31536000; includeSubDomains'
          }
        ]
      }
    ]
  }
}
```

#### 2. **Environment Variables**
```bash
# .env.local
DATABASE_URL=postgresql://...
JWT_SECRET=your-super-secret-key
NEXTAUTH_SECRET=your-nextauth-secret
SESSION_SECRET=your-session-secret
```

#### 3. **Rate Limiting (Production)**
```bash
npm install express-rate-limit
```

#### 4. **Database Security**
- ✅ Şifreler bcrypt ile hashlenmeli
- ✅ Hassas veriler encrypt edilmeli
- ✅ Database backupları düzenli alınmalı
- ✅ Least privilege principle

#### 5. **API Security**
```typescript
// API route koruması
export async function POST(req: Request) {
  // Auth kontrolü
  const session = await getSession(req)
  if (!session) {
    return new Response('Unauthorized', { status: 401 })
  }
  
  // Rate limiting
  const rateLimitResult = await checkRateLimit(session.user.id)
  if (!rateLimitResult.allowed) {
    return new Response('Too Many Requests', { status: 429 })
  }
  
  // CSRF kontrolü
  const csrfToken = req.headers.get('x-csrf-token')
  if (!verifyCSRFToken(csrfToken, session.csrfToken)) {
    return new Response('Invalid CSRF Token', { status: 403 })
  }
  
  // İşlem...
}
```

#### 6. **File Upload Security**
```typescript
// Dosya yükleme kontrolü
const allowedTypes = ['image/jpeg', 'image/png', 'image/webp']
const maxSize = 5 * 1024 * 1024 // 5MB

if (!allowedTypes.includes(file.type)) {
  throw new Error('Invalid file type')
}

if (file.size > maxSize) {
  throw new Error('File too large')
}

// Dosya adını sanitize et
const sanitizedFileName = file.name.replace(/[^a-zA-Z0-9.-]/g, '')
```

#### 7. **Headers Security**
```javascript
// next.config.js
module.exports = {
  async headers() {
    return [
      {
        source: '/:path*',
        headers: [
          {
            key: 'X-Frame-Options',
            value: 'DENY'
          },
          {
            key: 'X-Content-Type-Options',
            value: 'nosniff'
          },
          {
            key: 'X-XSS-Protection',
            value: '1; mode=block'
          },
          {
            key: 'Referrer-Policy',
            value: 'strict-origin-when-cross-origin'
          },
          {
            key: 'Permissions-Policy',
            value: 'camera=(), microphone=(), geolocation=()'
          }
        ]
      }
    ]
  }
}
```

### 📋 Güvenlik Kontrol Listesi

#### Pre-Production
- [ ] Tüm şifreler bcrypt ile hashlendi
- [ ] Environment variables ayarlandı
- [ ] HTTPS sertifikası kuruldu
- [ ] Rate limiting aktif
- [ ] CSRF koruması aktif
- [ ] Input validation tüm formlarda
- [ ] SQL injection koruması
- [ ] XSS koruması
- [ ] File upload güvenliği
- [ ] Security headers yapılandırıldı

#### Monitoring
- [ ] Audit log sistemi aktif
- [ ] Failed login attempts monitörleniyorr
- [ ] Anormal aktivite alertleri
- [ ] Database backup stratejisi
- [ ] Error logging (Sentry vb.)

#### Regular Maintenance
- [ ] Dependency güncellemeleri (npm audit)
- [ ] Security patch'ler
- [ ] Penetrasyon testleri
- [ ] Code review
- [ ] Password rotation policy

### 🚨 Güvenlik İhlali Durumunda

1. **Hemen Yapılacaklar**
   - Etkilenen hesapları kilitle
   - Session'ları iptal et
   - Şifreleri sıfırla
   - Logları incele

2. **Kullanıcıları Bilgilendir**
   - E-posta bildirimi
   - Dashboard uyarısı
   - Şifre değiştirme zorunluluğu

3. **Açığı Kapat**
   - Root cause analizi
   - Fix deploy et
   - Testleri güncelle

### 📞 İletişim

Güvenlik açığı bildirmek için:
- **Email**: security@dansdersleri.com
- **Response Time**: 24 saat içinde

---

**Son Güncelleme**: 4 Ekim 2024  
**Versiyon**: 1.0.0
