'use client'

import { useState } from 'react'
import { CheckCircle, XCircle, Eye, Clock, User, MapPin, Star } from 'lucide-react'

export default function PendingApproval() {
  const [pendingCourses, setPendingCourses] = useState([
    {
      id: 1,
      name: 'Salsa & Bachata Akademi',
      instructor: 'Mehmet Yılmaz',
      city: 'İstanbul',
      category: 'Salsa',
      level: 'Başlangıç',
      price: 450,
      submittedAt: '2024-10-03',
      status: 'pending',
      description: 'Profesyonel salsa ve bachata eğitimi...'
    },
    {
      id: 2,
      name: 'Tango Tutkusu',
      instructor: 'Ayşe Demir',
      city: 'Ankara',
      category: 'Tango',
      level: 'Orta',
      price: 600,
      submittedAt: '2024-10-02',
      status: 'pending',
      description: 'Arjantin tango eğitimi...'
    },
    {
      id: 3,
      name: 'Hip-Hop Gençlik',
      instructor: 'Can Özkan',
      city: 'İzmir',
      category: 'Hip-Hop',
      level: 'Tüm Seviyeler',
      price: 350,
      submittedAt: '2024-10-01',
      status: 'pending',
      description: 'Modern hip-hop dans eğitimi...'
    }
  ])

  const [selectedCourses, setSelectedCourses] = useState([])

  const handleApprove = (courseId) => {
    if (confirm('Bu kursu onaylamak istediğinizden emin misiniz?')) {
      setPendingCourses(pendingCourses.filter(course => course.id !== courseId))
      alert('Kurs onaylandı!')
    }
  }

  const handleReject = (courseId) => {
    if (confirm('Bu kursu reddetmek istediğinizden emin misiniz?')) {
      setPendingCourses(pendingCourses.filter(course => course.id !== courseId))
      alert('Kurs reddedildi!')
    }
  }

  const handleBulkApprove = () => {
    if (selectedCourses.length === 0) {
      alert('Lütfen onaylamak istediğiniz kursları seçin!')
      return
    }
    
    if (confirm(`${selectedCourses.length} kursu onaylamak istediğinizden emin misiniz?`)) {
      setPendingCourses(pendingCourses.filter(course => !selectedCourses.includes(course.id)))
      setSelectedCourses([])
      alert('Seçili kurslar onaylandı!')
    }
  }

  const toggleSelectCourse = (courseId) => {
    setSelectedCourses(prev => 
      prev.includes(courseId) 
        ? prev.filter(id => id !== courseId)
        : [...prev, courseId]
    )
  }

  return (
    <div className="space-y-6">
      {/* Page Header */}
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-3xl font-bold text-gray-900">Onay Bekleyen Kurslar</h1>
          <p className="text-gray-600 mt-2">Yeni eklenen kursları inceleyin ve onaylayın</p>
        </div>
        {selectedCourses.length > 0 && (
          <button
            onClick={handleBulkApprove}
            className="btn-primary flex items-center space-x-2"
          >
            <CheckCircle className="w-4 h-4" />
            <span>Toplu Onayla ({selectedCourses.length})</span>
          </button>
        )}
      </div>

      {/* Stats */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
        <div className="bg-white rounded-xl shadow-lg p-6">
          <div className="flex items-center">
            <div className="p-3 bg-yellow-100 rounded-lg">
              <Clock className="w-6 h-6 text-yellow-600" />
            </div>
            <div className="ml-4">
              <p className="text-sm font-medium text-gray-600">Bekleyen</p>
              <p className="text-2xl font-bold text-gray-900">{pendingCourses.length}</p>
            </div>
          </div>
        </div>

        <div className="bg-white rounded-xl shadow-lg p-6">
          <div className="flex items-center">
            <div className="p-3 bg-green-100 rounded-lg">
              <CheckCircle className="w-6 h-6 text-green-600" />
            </div>
            <div className="ml-4">
              <p className="text-sm font-medium text-gray-600">Bu Hafta Onaylanan</p>
              <p className="text-2xl font-bold text-gray-900">12</p>
            </div>
          </div>
        </div>

        <div className="bg-white rounded-xl shadow-lg p-6">
          <div className="flex items-center">
            <div className="p-3 bg-red-100 rounded-lg">
              <XCircle className="w-6 h-6 text-red-600" />
            </div>
            <div className="ml-4">
              <p className="text-sm font-medium text-gray-600">Bu Hafta Reddedilen</p>
              <p className="text-2xl font-bold text-gray-900">3</p>
            </div>
          </div>
        </div>
      </div>

      {/* Courses List */}
      <div className="bg-white rounded-xl shadow-lg overflow-hidden">
        <div className="overflow-x-auto">
          <table className="min-w-full divide-y divide-gray-200">
            <thead className="bg-gray-50">
              <tr>
                <th className="px-6 py-3 text-left">
                  <input
                    type="checkbox"
                    checked={selectedCourses.length === pendingCourses.length}
                    onChange={() => {
                      if (selectedCourses.length === pendingCourses.length) {
                        setSelectedCourses([])
                      } else {
                        setSelectedCourses(pendingCourses.map(course => course.id))
                      }
                    }}
                    className="rounded border-gray-300 text-primary-600 focus:ring-primary-500"
                  />
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Kurs Bilgileri
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Eğitmen
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Konum
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Fiyat
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  Tarih
                </th>
                <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                  İşlemler
                </th>
              </tr>
            </thead>
            <tbody className="bg-white divide-y divide-gray-200">
              {pendingCourses.map((course) => (
                <tr key={course.id} className="hover:bg-gray-50">
                  <td className="px-6 py-4 whitespace-nowrap">
                    <input
                      type="checkbox"
                      checked={selectedCourses.includes(course.id)}
                      onChange={() => toggleSelectCourse(course.id)}
                      className="rounded border-gray-300 text-primary-600 focus:ring-primary-500"
                    />
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <div>
                      <div className="text-sm font-medium text-gray-900">{course.name}</div>
                      <div className="text-sm text-gray-500">{course.category} • {course.level}</div>
                      <div className="text-xs text-gray-400 mt-1 line-clamp-2">{course.description}</div>
                    </div>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <div className="flex items-center">
                      <User className="w-4 h-4 text-gray-400 mr-2" />
                      <div className="text-sm text-gray-900">{course.instructor}</div>
                    </div>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <div className="flex items-center">
                      <MapPin className="w-4 h-4 text-gray-400 mr-2" />
                      <div className="text-sm text-gray-900">{course.city}</div>
                    </div>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <div className="text-sm font-medium text-gray-900">₺{course.price}</div>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap">
                    <div className="text-sm text-gray-900">{course.submittedAt}</div>
                  </td>
                  <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                    <div className="flex items-center space-x-2">
                      <button className="text-blue-600 hover:text-blue-900">
                        <Eye className="w-4 h-4" />
                      </button>
                      <button
                        onClick={() => handleApprove(course.id)}
                        className="text-green-600 hover:text-green-900"
                      >
                        <CheckCircle className="w-4 h-4" />
                      </button>
                      <button
                        onClick={() => handleReject(course.id)}
                        className="text-red-600 hover:text-red-900"
                      >
                        <XCircle className="w-4 h-4" />
                      </button>
                    </div>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      </div>

      {pendingCourses.length === 0 && (
        <div className="text-center py-12">
          <Clock className="w-12 h-12 text-gray-400 mx-auto mb-4" />
          <h3 className="text-lg font-medium text-gray-900 mb-2">Onay Bekleyen Kurs Yok</h3>
          <p className="text-gray-500">Şu anda onay bekleyen kurs bulunmuyor.</p>
        </div>
      )}
    </div>
  )
}

