'use client'

import { useState } from 'react'
import { BarChart3, TrendingUp, Users, School, Calendar, Download } from 'lucide-react'

export default function RaporlarPage() {
  const [selectedPeriod, setSelectedPeriod] = useState('thisMonth')

  const reportCategories = [
    {
      id: 1,
      name: 'Genel İstatistikler',
      description: 'Platform geneli istatistikler ve metrikler',
      icon: BarChart3,
      color: 'blue',
      reports: 5
    },
    {
      id: 2,
      name: 'Kurs Performansı',
      description: 'Kursların performans analizi ve karşılaştırmaları',
      icon: School,
      color: 'green',
      reports: 3
    },
    {
      id: 3,
      name: 'Gelir Raporları',
      description: 'Finansal performans ve gelir analizleri',
      icon: TrendingUp,
      color: 'purple',
      reports: 4
    },
    {
      id: 4,
      name: 'Kullanıcı Analizi',
      description: 'Kullanıcı davranışları ve segmentasyonu',
      icon: Users,
      color: 'orange',
      reports: 2
    }
  ]

  const recentReports = [
    {
      id: 1,
      name: 'Aylık Genel Rapor',
      category: 'Genel İstatistikler',
      generatedAt: '2024-03-15',
      status: 'Hazır'
    },
    {
      id: 2,
      name: 'Kurs Performans Analizi',
      category: 'Kurs Performansı',
      generatedAt: '2024-03-14',
      status: 'Hazır'
    },
    {
      id: 3,
      name: 'Gelir Trend Raporu',
      category: 'Gelir Raporları',
      generatedAt: '2024-03-13',
      status: 'Hazırlanıyor'
    }
  ]

  const getColorClasses = (color: string) => {
    switch (color) {
      case 'blue':
        return 'bg-blue-100 text-blue-600'
      case 'green':
        return 'bg-green-100 text-green-600'
      case 'purple':
        return 'bg-purple-100 text-purple-600'
      case 'orange':
        return 'bg-orange-100 text-orange-600'
      default:
        return 'bg-gray-100 text-gray-600'
    }
  }

  const getStatusColor = (status: string) => {
    switch (status) {
      case 'Hazır':
        return 'bg-green-100 text-green-800'
      case 'Hazırlanıyor':
        return 'bg-yellow-100 text-yellow-800'
      case 'Hata':
        return 'bg-red-100 text-red-800'
      default:
        return 'bg-gray-100 text-gray-800'
    }
  }

  return (
    <div className="space-y-6">
      {/* Header */}
      <div className="flex justify-between items-center">
        <div>
          <h1 className="text-2xl font-bold text-gray-900">Raporlar</h1>
          <p className="text-gray-600">Detaylı analizler ve raporlar oluşturun</p>
        </div>
        <div className="flex space-x-2">
          <select
            value={selectedPeriod}
            onChange={(e) => setSelectedPeriod(e.target.value)}
            className="px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent"
          >
            <option value="thisWeek">Bu Hafta</option>
            <option value="thisMonth">Bu Ay</option>
            <option value="lastMonth">Geçen Ay</option>
            <option value="thisYear">Bu Yıl</option>
          </select>
        </div>
      </div>

      {/* Quick Stats */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-6">
        <div className="bg-white p-6 rounded-lg shadow">
          <div className="flex items-center">
            <div className="p-2 bg-blue-100 rounded-lg">
              <BarChart3 className="w-6 h-6 text-blue-600" />
            </div>
            <div className="ml-4">
              <p className="text-sm font-medium text-gray-600">Toplam Rapor</p>
              <p className="text-2xl font-bold text-gray-900">24</p>
            </div>
          </div>
        </div>
        <div className="bg-white p-6 rounded-lg shadow">
          <div className="flex items-center">
            <div className="p-2 bg-green-100 rounded-lg">
              <div className="w-6 h-6 bg-green-600 rounded"></div>
            </div>
            <div className="ml-4">
              <p className="text-sm font-medium text-gray-600">Bu Ay Oluşturulan</p>
              <p className="text-2xl font-bold text-gray-900">8</p>
            </div>
          </div>
        </div>
        <div className="bg-white p-6 rounded-lg shadow">
          <div className="flex items-center">
            <div className="p-2 bg-purple-100 rounded-lg">
              <div className="w-6 h-6 bg-purple-600 rounded"></div>
            </div>
            <div className="ml-4">
              <p className="text-sm font-medium text-gray-600">Hazır Rapor</p>
              <p className="text-2xl font-bold text-gray-900">18</p>
            </div>
          </div>
        </div>
        <div className="bg-white p-6 rounded-lg shadow">
          <div className="flex items-center">
            <div className="p-2 bg-orange-100 rounded-lg">
              <div className="w-6 h-6 bg-orange-600 rounded"></div>
            </div>
            <div className="ml-4">
              <p className="text-sm font-medium text-gray-600">İndirme Sayısı</p>
              <p className="text-2xl font-bold text-gray-900">156</p>
            </div>
          </div>
        </div>
      </div>

      {/* Report Categories */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
        {reportCategories.map((category) => (
          <div key={category.id} className="bg-white rounded-lg shadow p-6 hover:shadow-lg transition-shadow">
            <div className="flex items-center mb-4">
              <div className={`p-2 rounded-lg ${getColorClasses(category.color)}`}>
                <category.icon className="w-6 h-6" />
              </div>
              <div className="ml-3">
                <h3 className="text-lg font-semibold text-gray-900">{category.name}</h3>
                <p className="text-sm text-gray-600">{category.reports} rapor</p>
              </div>
            </div>
            <p className="text-gray-600 text-sm mb-4">{category.description}</p>
            <button className="w-full bg-primary-600 hover:bg-primary-700 text-white py-2 px-4 rounded-lg">
              Raporları Görüntüle
            </button>
          </div>
        ))}
      </div>

      {/* Recent Reports */}
      <div className="bg-white rounded-lg shadow">
        <div className="px-6 py-4 border-b border-gray-200">
          <h3 className="text-lg font-semibold text-gray-900">Son Oluşturulan Raporlar</h3>
        </div>
        <div className="divide-y divide-gray-200">
          {recentReports.map((report) => (
            <div key={report.id} className="p-6 hover:bg-gray-50">
              <div className="flex items-center justify-between">
                <div className="flex-1">
                  <div className="flex items-center space-x-3 mb-2">
                    <h4 className="text-lg font-medium text-gray-900">{report.name}</h4>
                    <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getStatusColor(report.status)}`}>
                      {report.status}
                    </span>
                  </div>
                  <p className="text-gray-600 mb-2">{report.category}</p>
                  <div className="flex items-center space-x-4 text-sm text-gray-500">
                    <span className="flex items-center">
                      <Calendar className="w-4 h-4 mr-1" />
                      {new Date(report.generatedAt).toLocaleDateString('tr-TR')}
                    </span>
                  </div>
                </div>
                <div className="flex space-x-2">
                  <button className="bg-primary-600 hover:bg-primary-700 text-white px-4 py-2 rounded-lg flex items-center space-x-2">
                    <Download className="w-4 h-4" />
                    <span>İndir</span>
                  </button>
                  <button className="bg-gray-600 hover:bg-gray-700 text-white px-4 py-2 rounded-lg flex items-center space-x-2">
                    <BarChart3 className="w-4 h-4" />
                    <span>Görüntüle</span>
                  </button>
                </div>
              </div>
            </div>
          ))}
        </div>
      </div>

      {/* Generate New Report */}
      <div className="bg-white p-6 rounded-lg shadow">
        <h3 className="text-lg font-semibold text-gray-900 mb-4">Yeni Rapor Oluştur</h3>
        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
          <button className="p-4 border-2 border-dashed border-gray-300 rounded-lg hover:border-primary-500 hover:bg-primary-50 transition-colors">
            <div className="text-center">
              <BarChart3 className="w-8 h-8 text-gray-400 mx-auto mb-2" />
              <h4 className="font-medium text-gray-900">Genel Rapor</h4>
              <p className="text-sm text-gray-600">Platform geneli analiz</p>
            </div>
          </button>
          <button className="p-4 border-2 border-dashed border-gray-300 rounded-lg hover:border-primary-500 hover:bg-primary-50 transition-colors">
            <div className="text-center">
              <School className="w-8 h-8 text-gray-400 mx-auto mb-2" />
              <h4 className="font-medium text-gray-900">Kurs Raporu</h4>
              <p className="text-sm text-gray-600">Kurs performans analizi</p>
            </div>
          </button>
          <button className="p-4 border-2 border-dashed border-gray-300 rounded-lg hover:border-primary-500 hover:bg-primary-50 transition-colors">
            <div className="text-center">
              <TrendingUp className="w-8 h-8 text-gray-400 mx-auto mb-2" />
              <h4 className="font-medium text-gray-900">Gelir Raporu</h4>
              <p className="text-sm text-gray-600">Finansal analiz</p>
            </div>
          </button>
        </div>
      </div>
    </div>
  )
}

