import { Suspense } from 'react'
import SearchHeader from '@/components/search/SearchHeader'
import SearchFilters from '@/components/search/SearchFilters'
import SearchResults from '@/components/search/SearchResults'
import SearchPagination from '@/components/search/SearchPagination'

interface SearchParams {
  q?: string
  city?: string
  type?: string
  level?: string
  price_min?: string
  price_max?: string
  rating?: string
  page?: string
}

interface SearchPageProps {
  searchParams: SearchParams
}

export async function generateMetadata({ searchParams }: SearchPageProps) {
  const { q, city, type } = searchParams
  
  let title = 'Dans Kursu Arama'
  let description = 'Türkiye genelinde dans kurslarını arayın ve keşfedin.'
  
  if (q || city || type) {
    const parts = []
    if (q) parts.push(q)
    if (type) parts.push(type)
    if (city) parts.push(city)
    
    title = `${parts.join(' ')} - Dans Kursu Arama | Dans Dersleri`
    description = `${parts.join(' ')} için dans kursları. En iyi fiyatlar ve kaliteli eğitim.`
  }

  return {
    title,
    description,
    openGraph: {
      title,
      description,
    },
  }
}

export default function SearchPage({ searchParams }: SearchPageProps) {
  return (
    <div className="min-h-screen bg-gray-50">
      <SearchHeader searchParams={searchParams} />
      
      <div className="container-custom py-8">
        <div className="grid grid-cols-1 lg:grid-cols-4 gap-8">
          {/* Filters Sidebar */}
          <div className="lg:col-span-1">
            <SearchFilters searchParams={searchParams} />
          </div>

          {/* Results */}
          <div className="lg:col-span-3">
            <Suspense fallback={<SearchResultsSkeleton />}>
              <SearchResults searchParams={searchParams} />
            </Suspense>
            
            <div className="mt-8">
              <Suspense fallback={<div className="h-12" />}>
                <SearchPagination searchParams={searchParams} />
              </Suspense>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}

function SearchResultsSkeleton() {
  return (
    <div className="space-y-6">
      {[...Array(6)].map((_, i) => (
        <div key={i} className="bg-white rounded-xl shadow-lg p-6 animate-pulse">
          <div className="flex space-x-4">
            <div className="w-32 h-32 bg-gray-200 rounded-lg"></div>
            <div className="flex-1 space-y-3">
              <div className="h-6 bg-gray-200 rounded w-3/4"></div>
              <div className="h-4 bg-gray-200 rounded w-1/2"></div>
              <div className="h-4 bg-gray-200 rounded w-2/3"></div>
              <div className="flex space-x-2">
                <div className="h-6 bg-gray-200 rounded w-16"></div>
                <div className="h-6 bg-gray-200 rounded w-16"></div>
              </div>
            </div>
          </div>
        </div>
      ))}
    </div>
  )
}

