'use client'

import { useState } from 'react'
import Link from 'next/link'
import { usePathname } from 'next/navigation'
import { 
  LayoutDashboard, 
  School, 
  Users, 
  MessageSquare, 
  DollarSign, 
  Settings,
  ChevronDown,
  ChevronRight,
  MapPin,
  BarChart3,
  FileText,
  Tag,
  Package
} from 'lucide-react'

interface MenuItem {
  name: string
  href: string
  iconName: string
  badge?: string
  subItems?: {
    name: string
    href: string
  }[]
}

export default function AdminSidebar() {
  const pathname = usePathname()
  const [expandedItems, setExpandedItems] = useState<string[]>(['Kurslar'])

  const getIcon = (iconName: string) => {
    const iconClassName = "w-5 h-5"
    switch (iconName) {
      case 'dashboard': return <LayoutDashboard className={iconClassName} />
      case 'school': return <School className={iconClassName} />
      case 'users': return <Users className={iconClassName} />
      case 'message': return <MessageSquare className={iconClassName} />
      case 'dollar': return <DollarSign className={iconClassName} />
      case 'map': return <MapPin className={iconClassName} />
      case 'chart': return <BarChart3 className={iconClassName} />
      case 'file': return <FileText className={iconClassName} />
      case 'settings': return <Settings className={iconClassName} />
      default: return null
    }
  }

  const menuItems: MenuItem[] = [
    {
      name: 'Dashboard',
      href: '/admin',
      iconName: 'dashboard'
    },
    {
      name: 'Kurslar',
      href: '/admin/kurslar',
      iconName: 'school',
      badge: '45',
      subItems: [
        { name: 'Tüm Kurslar', href: '/admin/kurslar' },
        { name: 'Yeni Kurs Ekle', href: '/admin/kurslar/yeni' },
        { name: 'Kategoriler', href: '/admin/kurslar/kategoriler' },
        { name: 'Onay Bekleyenler', href: '/admin/kurslar/onay' }
      ]
    },
    {
      name: 'Kullanıcılar',
      href: '/admin/kullanicilar',
      iconName: 'users',
      subItems: [
        { name: 'Tüm Kullanıcılar', href: '/admin/kullanicilar' },
        { name: 'Kurs Sahipleri', href: '/admin/kullanicilar/kurs-sahipleri' },
        { name: 'Üyelik Paketleri', href: '/admin/kullanicilar/paketler' }
      ]
    },
    {
      name: 'Değerlendirmeler',
      href: '/admin/degerlendirmeler',
      iconName: 'message',
      badge: '12'
    },
    {
      name: 'Finansal',
      href: '/admin/finansal',
      iconName: 'dollar',
      subItems: [
        { name: 'Ödemeler', href: '/admin/finansal/odemeler' },
        { name: 'Abonelikler', href: '/admin/finansal/abonelikler' },
        { name: 'Faturalar', href: '/admin/finansal/faturalar' },
        { name: 'Raporlar', href: '/admin/finansal/raporlar' }
      ]
    },
    {
      name: 'Şehirler',
      href: '/admin/sehirler',
      iconName: 'map'
    },
    {
      name: 'Raporlar',
      href: '/admin/raporlar',
      iconName: 'chart',
      subItems: [
        { name: 'Genel İstatistikler', href: '/admin/raporlar' },
        { name: 'Kurs Performansı', href: '/admin/raporlar/kurslar' },
        { name: 'Gelir Raporları', href: '/admin/raporlar/gelir' }
      ]
    },
    {
      name: 'İçerik',
      href: '/admin/icerik',
      iconName: 'file',
      subItems: [
        { name: 'Blog Yazıları', href: '/admin/icerik/blog' },
        { name: 'Sayfalar', href: '/admin/icerik/sayfalar' },
        { name: 'SSS', href: '/admin/icerik/sss' }
      ]
    },
    {
      name: 'Ayarlar',
      href: '/admin/ayarlar',
      iconName: 'settings',
      subItems: [
        { name: 'Genel Ayarlar', href: '/admin/ayarlar' },
        { name: 'E-posta Şablonları', href: '/admin/ayarlar/eposta' },
        { name: 'SEO Ayarları', href: '/admin/ayarlar/seo' },
        { name: 'Ödeme Ayarları', href: '/admin/ayarlar/odeme' }
      ]
    }
  ]

  const toggleExpand = (itemName: string) => {
    setExpandedItems(prev => 
      prev.includes(itemName) 
        ? prev.filter(name => name !== itemName)
        : [...prev, itemName]
    )
  }

  const isActive = (href: string) => {
    if (href === '/admin') {
      return pathname === href
    }
    return pathname.startsWith(href)
  }

  return (
    <aside className="w-64 bg-gray-900 text-white overflow-y-auto">
      {/* Logo */}
      <div className="p-6 border-b border-gray-800">
        <Link href="/admin" className="flex items-center space-x-2">
          <div className="w-10 h-10 bg-gradient-primary rounded-lg flex items-center justify-center">
            <span className="text-white font-bold text-xl">D</span>
          </div>
          <div>
            <h1 className="text-lg font-bold">Dans Dersleri</h1>
            <p className="text-xs text-gray-400">Admin Panel</p>
          </div>
        </Link>
      </div>

      {/* Navigation */}
      <nav className="p-4">
        <ul className="space-y-1">
          {menuItems.map((item) => (
            <li key={item.name}>
              <div>
                <Link
                  href={item.href}
                  onClick={(e) => {
                    if (item.subItems) {
                      e.preventDefault()
                      toggleExpand(item.name)
                    }
                  }}
                  className={`flex items-center justify-between px-4 py-3 rounded-lg transition-colors ${
                    isActive(item.href)
                      ? 'bg-primary-600 text-white'
                      : 'text-gray-300 hover:bg-gray-800'
                  }`}
                >
                  <div className="flex items-center space-x-3">
                    {getIcon(item.iconName)}
                    <span className="font-medium">{item.name}</span>
                  </div>
                  <div className="flex items-center space-x-2">
                    {item.badge && (
                      <span className="bg-primary-500 text-white text-xs px-2 py-0.5 rounded-full">
                        {item.badge}
                      </span>
                    )}
                    {item.subItems && (
                      expandedItems.includes(item.name) ? (
                        <ChevronDown className="w-4 h-4" />
                      ) : (
                        <ChevronRight className="w-4 h-4" />
                      )
                    )}
                  </div>
                </Link>

                {/* Sub Items */}
                {item.subItems && expandedItems.includes(item.name) && (
                  <ul className="ml-4 mt-1 space-y-1">
                    {item.subItems.map((subItem) => (
                      <li key={subItem.href}>
                        <Link
                          href={subItem.href}
                          className={`block px-4 py-2 rounded-lg text-sm transition-colors ${
                            pathname === subItem.href
                              ? 'bg-gray-800 text-white'
                              : 'text-gray-400 hover:bg-gray-800 hover:text-white'
                          }`}
                        >
                          {subItem.name}
                        </Link>
                      </li>
                    ))}
                  </ul>
                )}
              </div>
            </li>
          ))}
        </ul>
      </nav>

      {/* Bottom Info */}
      <div className="p-4 border-t border-gray-800 mt-auto">
        <div className="bg-gray-800 rounded-lg p-4">
          <h3 className="text-sm font-semibold mb-2">Yardıma mı ihtiyacınız var?</h3>
          <p className="text-xs text-gray-400 mb-3">
            Dokümantasyon ve destek için tıklayın
          </p>
          <Link
            href="/admin/yardim"
            className="block w-full bg-primary-600 hover:bg-primary-700 text-white text-sm text-center py-2 rounded-lg transition-colors"
          >
            Yardım Merkezi
          </Link>
        </div>
      </div>
    </aside>
  )
}
