'use client'

import { createContext, useContext, useEffect, useState } from 'react'
import { useRouter, usePathname } from 'next/navigation'
import type { User, Session } from '@/lib/auth'
import { validateSession, hasPermission } from '@/lib/auth'

interface AuthContextType {
  user: User | null
  session: Session | null
  loading: boolean
  login: (email: string, password: string) => Promise<void>
  logout: () => void
  checkPermission: (permission: string) => boolean
}

const AuthContext = createContext<AuthContextType | undefined>(undefined)

export function AuthProvider({ children }: { children: React.ReactNode }) {
  const [session, setSession] = useState<Session | null>(null)
  const [loading, setLoading] = useState(true)
  const router = useRouter()
  const pathname = usePathname()

  useEffect(() => {
    // Session kontrolü
    checkSession()
  }, [])

  const checkSession = () => {
    try {
      const stored = localStorage.getItem('adminSession')
      if (stored) {
        const sessionData: Session = JSON.parse(stored)
        
        if (validateSession(sessionData)) {
          setSession(sessionData)
        } else {
          // Session süresi dolmuş
          logout()
        }
      }
    } catch (error) {
      console.error('Session kontrolü hatası:', error)
      logout()
    } finally {
      setLoading(false)
    }
  }

  const login = async (email: string, password: string) => {
    // Bu fonksiyon LoginForm'dan çağrılacak
    // Gerçek API implementasyonu eklenecek
  }

  const logout = () => {
    localStorage.removeItem('adminSession')
    setSession(null)
    router.push('/admin/login')
  }

  const checkPermission = (permission: string): boolean => {
    if (!session?.user) return false
    return hasPermission(session.user.role, permission)
  }

  // Protected route kontrolü
  useEffect(() => {
    if (!loading && !session && pathname?.startsWith('/admin') && pathname !== '/admin/login') {
      router.push('/admin/login')
    }
  }, [session, loading, pathname, router])

  return (
    <AuthContext.Provider 
      value={{ 
        user: session?.user || null, 
        session, 
        loading, 
        login, 
        logout,
        checkPermission 
      }}
    >
      {children}
    </AuthContext.Provider>
  )
}

export function useAuth() {
  const context = useContext(AuthContext)
  if (context === undefined) {
    throw new Error('useAuth must be used within an AuthProvider')
  }
  return context
}

