'use client'

import { useState } from 'react'
import { Phone, MessageCircle, Heart, Share2, Star, Clock, Users, MapPin } from 'lucide-react'

interface Course {
  id: string
  name: string
  location: string
  price: number
  currency: string
  rating: number
  reviewCount: number
  duration: string
  level: string
  schedule: string
  contact: {
    phone: string
    email: string
    website: string
    whatsapp: string
  }
}

interface CourseCTAProps {
  course: Course
}

export default function CourseCTA({ course }: CourseCTAProps) {
  const [isFavorite, setIsFavorite] = useState(false)
  const [showContactForm, setShowContactForm] = useState(false)
  const [formData, setFormData] = useState({
    name: '',
    phone: '',
    email: '',
    message: ''
  })

  const handleSubmit = (e: React.FormEvent) => {
    e.preventDefault()
    // Form submission logic here
    console.log('Form submitted:', formData)
    alert('Mesajınız gönderildi! En kısa sürede size dönüş yapacağız.')
    setShowContactForm(false)
    setFormData({ name: '', phone: '', email: '', message: '' })
  }

  const shareCourse = async () => {
    if (navigator.share) {
      try {
        await navigator.share({
          title: course.name,
          text: course.name + ' - ' + course.location,
          url: window.location.href,
        })
      } catch (err) {
        console.log('Paylaşım iptal edildi')
      }
    } else {
      navigator.clipboard.writeText(window.location.href)
      alert('Link kopyalandı!')
    }
  }

  return (
    <div className="sticky top-6 space-y-6">
      {/* Course Info Card */}
      <div className="bg-white rounded-xl shadow-lg p-6">
        <div className="text-center mb-6">
          <div className="text-3xl font-bold text-primary-600 mb-2">
            {course.price} {course.currency}
          </div>
          <div className="text-sm text-gray-600">aylık ücret</div>
        </div>

        {/* Course Details */}
        <div className="space-y-3 mb-6">
          <div className="flex items-center justify-between text-sm">
            <div className="flex items-center space-x-2">
              <Star className="w-4 h-4 text-yellow-400 fill-current" />
              <span className="text-gray-600">Puan</span>
            </div>
            <span className="font-semibold">{course.rating} ({course.reviewCount})</span>
          </div>
          
          <div className="flex items-center justify-between text-sm">
            <div className="flex items-center space-x-2">
              <Clock className="w-4 h-4 text-gray-600" />
              <span className="text-gray-600">Süre</span>
            </div>
            <span className="font-semibold">{course.duration}</span>
          </div>
          
          <div className="flex items-center justify-between text-sm">
            <div className="flex items-center space-x-2">
              <Users className="w-4 h-4 text-gray-600" />
              <span className="text-gray-600">Seviye</span>
            </div>
            <span className="font-semibold">{course.level}</span>
          </div>
          
          <div className="flex items-center justify-between text-sm">
            <div className="flex items-center space-x-2">
              <MapPin className="w-4 h-4 text-gray-600" />
              <span className="text-gray-600">Konum</span>
            </div>
            <span className="font-semibold text-right">{course.location}</span>
          </div>
        </div>

        {/* Action Buttons */}
        <div className="space-y-3">
          <a
            href={`tel:${course.contact.phone}`}
            className="w-full btn-primary flex items-center justify-center space-x-2"
          >
            <Phone className="w-4 h-4" />
            <span>Hemen Ara</span>
          </a>
          
          <a
            href={`https://wa.me/${course.contact.whatsapp.replace(/[^0-9]/g, '')}`}
            target="_blank"
            rel="noopener noreferrer"
            className="w-full bg-green-600 hover:bg-green-700 text-white font-medium py-3 px-4 rounded-lg transition-colors duration-200 flex items-center justify-center space-x-2"
          >
            <MessageCircle className="w-4 h-4" />
            <span>WhatsApp</span>
          </a>
          
          <button
            onClick={() => setShowContactForm(true)}
            className="w-full btn-outline"
          >
            Bilgi Al
          </button>
        </div>

        {/* Social Actions */}
        <div className="flex items-center justify-center space-x-4 mt-6 pt-6 border-t border-gray-200">
          <button
            onClick={() => setIsFavorite(!isFavorite)}
            aria-label={isFavorite ? "Favorilerden çıkar" : "Favorilere ekle"}
            className={`p-2 rounded-full transition-colors min-h-[44px] min-w-[44px] ${
              isFavorite 
                ? 'bg-red-100 text-red-600' 
                : 'bg-gray-100 text-gray-600 hover:bg-gray-200'
            }`}
          >
            <Heart className={`w-5 h-5 ${isFavorite ? 'fill-current' : ''}`} aria-hidden="true" />
          </button>
          
          <button
            onClick={shareCourse}
            aria-label="Kursu paylaş"
            className="p-2 rounded-full bg-gray-100 text-gray-600 hover:bg-gray-200 transition-colors min-h-[44px] min-w-[44px]"
          >
            <Share2 className="w-5 h-5" aria-hidden="true" />
          </button>
        </div>
      </div>

      {/* Schedule Info */}
      <div className="bg-white rounded-xl shadow-lg p-6">
        <h3 className="font-semibold text-gray-900 mb-3">Ders Programı</h3>
        <p className="text-sm text-gray-600">{course.schedule}</p>
      </div>

      {/* Contact Form Modal */}
      {showContactForm && (
        <div className="fixed inset-0 bg-black/50 flex items-center justify-center p-4 z-50">
          <div className="bg-white rounded-xl p-6 w-full max-w-md">
            <h3 className="text-xl font-bold text-gray-900 mb-4">Bilgi Formu</h3>
            
            <form onSubmit={handleSubmit} className="space-y-4">
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  Ad Soyad
                </label>
                <input
                  type="text"
                  required
                  value={formData.name}
                  onChange={(e) => setFormData({...formData, name: e.target.value})}
                  className="w-full input-field"
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  Telefon
                </label>
                <input
                  type="tel"
                  required
                  value={formData.phone}
                  onChange={(e) => setFormData({...formData, phone: e.target.value})}
                  className="w-full input-field"
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  E-posta
                </label>
                <input
                  type="email"
                  value={formData.email}
                  onChange={(e) => setFormData({...formData, email: e.target.value})}
                  className="w-full input-field"
                />
              </div>
              
              <div>
                <label className="block text-sm font-medium text-gray-700 mb-1">
                  Mesaj
                </label>
                <textarea
                  rows={3}
                  value={formData.message}
                  onChange={(e) => setFormData({...formData, message: e.target.value})}
                  className="w-full input-field"
                  placeholder="Sormak istediğiniz sorular..."
                />
              </div>
              
              <div className="flex space-x-3">
                <button
                  type="button"
                  onClick={() => setShowContactForm(false)}
                  className="flex-1 btn-outline"
                >
                  İptal
                </button>
                <button
                  type="submit"
                  className="flex-1 btn-primary"
                >
                  Gönder
                </button>
              </div>
            </form>
          </div>
        </div>
      )}
    </div>
  )
}
