'use client'

import { useState } from 'react'
import { Star, MapPin, Clock, Users, Heart, Share2, Phone, MessageCircle } from 'lucide-react'
import Image from 'next/image'

interface Course {
  id: string
  name: string
  location: string
  address: string
  danceTypes: string[]
  rating: number
  reviewCount: number
  price: number
  currency: string
  duration: string
  level: string
  schedule: string
  image: string
  gallery: string[]
  description: string
  features: string[]
  contact: {
    phone: string
    email: string
    website: string
    whatsapp: string
  }
}

interface CourseHeaderProps {
  course: Course
}

export default function CourseHeader({ course }: CourseHeaderProps) {
  const [currentImageIndex, setCurrentImageIndex] = useState(0)
  const [isFavorite, setIsFavorite] = useState(false)

  const allImages = [course.image, ...course.gallery]

  const nextImage = () => {
    setCurrentImageIndex((prev) => (prev + 1) % allImages.length)
  }

  const prevImage = () => {
    setCurrentImageIndex((prev) => (prev - 1 + allImages.length) % allImages.length)
  }

  const shareCourse = async () => {
    if (navigator.share) {
      try {
        await navigator.share({
          title: course.name,
          text: course.description,
          url: window.location.href,
        })
      } catch (err) {
        console.log('Paylaşım iptal edildi')
      }
    } else {
      // Fallback: URL'yi kopyala
      navigator.clipboard.writeText(window.location.href)
      alert('Link kopyalandı!')
    }
  }

  return (
    <div className="bg-white">
      {/* Image Gallery */}
      <div className="relative h-96 md:h-[500px] overflow-hidden">
        <Image
          src={allImages[currentImageIndex]}
          alt={course.name}
          fill
          className="object-cover"
          priority
        />
        
        {/* Overlay */}
        <div className="absolute inset-0 bg-gradient-to-t from-black/60 via-transparent to-transparent"></div>
        
        {/* Navigation Arrows */}
        {allImages.length > 1 && (
          <>
            <button
              onClick={prevImage}
              aria-label="Önceki fotoğraf"
              className="absolute left-4 top-1/2 transform -translate-y-1/2 w-12 h-12 bg-white/90 rounded-full flex items-center justify-center hover:bg-white transition-colors"
            >
              <span className="text-2xl" aria-hidden="true">‹</span>
            </button>
            <button
              onClick={nextImage}
              aria-label="Sonraki fotoğraf"
              className="absolute right-4 top-1/2 transform -translate-y-1/2 w-12 h-12 bg-white/90 rounded-full flex items-center justify-center hover:bg-white transition-colors"
            >
              <span className="text-2xl" aria-hidden="true">›</span>
            </button>
          </>
        )}

        {/* Image Indicators */}
        {allImages.length > 1 && (
          <div className="absolute bottom-4 left-1/2 transform -translate-x-1/2 flex space-x-2">
            {allImages.map((_, index) => (
              <button
                key={index}
                onClick={() => setCurrentImageIndex(index)}
                aria-label={`Fotoğraf ${index + 1}`}
                className={`w-3 h-3 rounded-full transition-colors ${
                  index === currentImageIndex ? 'bg-white' : 'bg-white/50'
                }`}
              />
            ))}
          </div>
        )}

        {/* Action Buttons */}
        <div className="absolute top-4 right-4 flex space-x-2">
          <button
            onClick={() => setIsFavorite(!isFavorite)}
            aria-label={isFavorite ? "Favorilerden çıkar" : "Favorilere ekle"}
            className="w-12 h-12 bg-white/90 rounded-full flex items-center justify-center hover:bg-white transition-colors"
          >
            <Heart className={`w-5 h-5 ${isFavorite ? 'text-red-500 fill-current' : 'text-gray-600'}`} aria-hidden="true" />
          </button>
          <button
            onClick={shareCourse}
            aria-label="Kursu paylaş"
            className="w-12 h-12 bg-white/90 rounded-full flex items-center justify-center hover:bg-white transition-colors"
          >
            <Share2 className="w-5 h-5 text-gray-600" aria-hidden="true" />
          </button>
        </div>

        {/* Price Badge */}
        <div className="absolute bottom-4 left-4">
          <div className="bg-primary-600 text-white px-4 py-2 rounded-lg text-lg font-bold">
            {course.price} {course.currency}/ay
          </div>
        </div>
      </div>

      {/* Course Info */}
      <div className="p-6 md:p-8">
        <div className="flex flex-col lg:flex-row lg:items-start lg:justify-between gap-6">
          {/* Main Info */}
          <div className="flex-1">
            <h1 className="text-3xl md:text-4xl font-bold text-gray-900 mb-4">
              {course.name}
            </h1>
            
            <div className="flex items-center text-gray-600 mb-4">
              <MapPin className="w-5 h-5 mr-2" />
              <span className="text-lg">{course.location}</span>
            </div>

            <p className="text-gray-700 text-lg mb-6">
              {course.description}
            </p>

            {/* Dance Types */}
            <div className="flex flex-wrap gap-2 mb-6">
              {course.danceTypes.map((type) => (
                <span
                  key={type}
                  className="bg-primary-100 text-primary-800 px-3 py-1 rounded-full text-sm font-medium"
                >
                  {type}
                </span>
              ))}
            </div>

            {/* Features */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-2 mb-6">
              {course.features.map((feature) => (
                <div key={feature} className="flex items-center text-gray-600">
                  <div className="w-2 h-2 bg-primary-500 rounded-full mr-3"></div>
                  {feature}
                </div>
              ))}
            </div>
          </div>

          {/* Side Info */}
          <div className="lg:w-80 space-y-4">
            {/* Rating & Reviews */}
            <div className="bg-gray-50 rounded-lg p-4">
              <div className="flex items-center justify-between mb-2">
                <div className="flex items-center space-x-1">
                  <Star className="w-5 h-5 text-yellow-400 fill-current" />
                  <span className="text-2xl font-bold">{course.rating}</span>
                </div>
                <span className="text-gray-600">({course.reviewCount} değerlendirme)</span>
              </div>
              <div className="text-sm text-gray-600">
                {course.level} seviye • {course.duration}
              </div>
            </div>

            {/* Schedule */}
            <div className="bg-gray-50 rounded-lg p-4">
              <div className="flex items-center space-x-2 mb-2">
                <Clock className="w-5 h-5 text-gray-600" />
                <span className="font-semibold">Ders Programı</span>
              </div>
              <div className="text-sm text-gray-600">
                {course.schedule}
              </div>
            </div>

            {/* Contact Buttons */}
            <div className="space-y-2">
              <a
                href={`tel:${course.contact.phone}`}
                className="w-full btn-primary flex items-center justify-center space-x-2"
              >
                <Phone className="w-4 h-4" />
                <span>Ara</span>
              </a>
              <a
                href={`https://wa.me/${course.contact.whatsapp.replace(/[^0-9]/g, '')}`}
                target="_blank"
                rel="noopener noreferrer"
                className="w-full bg-green-600 hover:bg-green-700 text-white font-medium py-3 px-4 rounded-lg transition-colors duration-200 flex items-center justify-center space-x-2"
              >
                <MessageCircle className="w-4 h-4" />
                <span>WhatsApp</span>
              </a>
            </div>
          </div>
        </div>
      </div>
    </div>
  )
}
