'use client'

import { useState } from 'react'
import { MapPin, Navigation, Clock, Car, Bus, Train, Phone, MessageCircle } from 'lucide-react'

interface Location {
  address: string
  coordinates: { lat: number; lng: number }
  district: string
  city: string
  transportation: string
}

interface CourseLocationProps {
  location: Location
}

export default function CourseLocation({ location }: CourseLocationProps) {
  const [mapLoaded, setMapLoaded] = useState(false)

  const openInMaps = () => {
    const url = `https://www.google.com/maps?q=${location.coordinates.lat},${location.coordinates.lng}`
    window.open(url, '_blank')
  }

  const getDirections = () => {
    const url = `https://www.google.com/maps/dir/?api=1&destination=${location.coordinates.lat},${location.coordinates.lng}`
    window.open(url, '_blank')
  }

  return (
    <div className="bg-white rounded-xl shadow-lg p-6 md:p-8">
      <h2 className="text-2xl font-bold text-gray-900 mb-6">Konum & Ulaşım</h2>
      
      <div className="grid grid-cols-1 lg:grid-cols-2 gap-8">
        {/* Address Info */}
        <div className="space-y-6">
          <div>
            <h3 className="text-lg font-semibold text-gray-900 mb-3">Adres Bilgileri</h3>
            <div className="space-y-3">
              <div className="flex items-start space-x-3">
                <MapPin className="w-5 h-5 text-primary-600 mt-1 flex-shrink-0" />
                <div>
                  <p className="text-gray-700 font-medium">{location.address}</p>
                  <p className="text-gray-600 text-sm">{location.district}, {location.city}</p>
                </div>
              </div>
            </div>
          </div>

          {/* Transportation */}
          <div>
            <h3 className="text-lg font-semibold text-gray-900 mb-3">Ulaşım Bilgileri</h3>
            <div className="space-y-3">
              <div className="flex items-start space-x-3">
                <Bus className="w-5 h-5 text-blue-600 mt-1 flex-shrink-0" />
                <div>
                  <p className="text-gray-700 font-medium">Toplu Taşıma</p>
                  <p className="text-gray-600 text-sm">{location.transportation}</p>
                </div>
              </div>
              
              <div className="flex items-start space-x-3">
                <Car className="w-5 h-5 text-green-600 mt-1 flex-shrink-0" />
                <div>
                  <p className="text-gray-700 font-medium">Araç ile Ulaşım</p>
                  <p className="text-gray-600 text-sm">Ücretsiz otopark mevcuttur</p>
                </div>
              </div>
            </div>
          </div>

          {/* Action Buttons */}
          <div className="space-y-3">
            <button
              onClick={openInMaps}
              className="w-full btn-primary flex items-center justify-center space-x-2"
            >
              <MapPin className="w-4 h-4" />
              <span>Haritada Görüntüle</span>
            </button>
            <button
              onClick={getDirections}
              className="w-full btn-outline flex items-center justify-center space-x-2"
            >
              <Navigation className="w-4 h-4" />
              <span>Yol Tarifi Al</span>
            </button>
          </div>
        </div>

        {/* Map Placeholder */}
        <div className="space-y-4">
          <h3 className="text-lg font-semibold text-gray-900">Harita</h3>
          <div className="relative h-64 bg-gray-100 rounded-lg overflow-hidden">
            {!mapLoaded && (
              <div className="absolute inset-0 flex items-center justify-center bg-gray-100">
                <div className="text-center">
                  <MapPin className="w-12 h-12 text-gray-400 mx-auto mb-2" />
                  <p className="text-gray-500">Harita yükleniyor...</p>
                </div>
              </div>
            )}
            
            {/* Google Maps Embed */}
            <iframe
              src={`https://www.google.com/maps/embed/v1/place?key=YOUR_API_KEY&q=${location.coordinates.lat},${location.coordinates.lng}`}
              width="100%"
              height="100%"
              style={{ border: 0 }}
              allowFullScreen
              loading="lazy"
              referrerPolicy="no-referrer-when-downgrade"
              onLoad={() => setMapLoaded(true)}
              className="absolute inset-0"
            />
          </div>

          {/* Quick Contact */}
          <div className="bg-primary-50 rounded-lg p-4">
            <h4 className="font-semibold text-gray-900 mb-3">Hızlı İletişim</h4>
            <div className="space-y-2">
              <a
                href="tel:+905321234567"
                className="flex items-center space-x-2 text-primary-600 hover:text-primary-700"
              >
                <Phone className="w-4 h-4" />
                <span className="text-sm">+90 532 123 4567</span>
              </a>
              <a
                href="https://wa.me/905321234567"
                target="_blank"
                rel="noopener noreferrer"
                className="flex items-center space-x-2 text-green-600 hover:text-green-700"
              >
                <MessageCircle className="w-4 h-4" />
                <span className="text-sm">WhatsApp</span>
              </a>
            </div>
          </div>
        </div>
      </div>

      {/* Additional Info */}
      <div className="mt-8 p-4 bg-gray-50 rounded-lg">
        <div className="flex items-start space-x-3">
          <Clock className="w-5 h-5 text-primary-600 mt-1 flex-shrink-0" />
          <div>
            <h4 className="font-semibold text-gray-900 mb-1">Önemli Notlar</h4>
            <ul className="text-sm text-gray-600 space-y-1">
              <li>• Kurs binasına 5 dakika önce gelmeniz önerilir</li>
              <li>• Otopark ücretsizdir ve 24 saat açıktır</li>
              <li>• Metro istasyonundan 5 dakika yürüme mesafesindedir</li>
              <li>• Engelli erişimi mevcuttur</li>
            </ul>
          </div>
        </div>
      </div>
    </div>
  )
}

