'use client'

import { useState } from 'react'
import { Search, Filter, MapPin, X } from 'lucide-react'
import { useRouter, useSearchParams } from 'next/navigation'

interface SearchParams {
  q?: string
  city?: string
  type?: string
  level?: string
  price_min?: string
  price_max?: string
  rating?: string
  page?: string
}

interface SearchHeaderProps {
  searchParams: SearchParams
}

export default function SearchHeader({ searchParams }: SearchHeaderProps) {
  const router = useRouter()
  const urlSearchParams = useSearchParams()
  const [searchQuery, setSearchQuery] = useState(searchParams.q || '')
  const [showMobileFilters, setShowMobileFilters] = useState(false)

  const handleSearch = (e: React.FormEvent) => {
    e.preventDefault()
    const params = new URLSearchParams(urlSearchParams)
    params.set('q', searchQuery)
    params.delete('page') // Reset to first page
    router.push(`/arama?${params.toString()}`)
  }

  const clearFilters = () => {
    router.push('/arama')
  }

  const getActiveFiltersCount = () => {
    let count = 0
    if (searchParams.city) count++
    if (searchParams.type) count++
    if (searchParams.level) count++
    if (searchParams.price_min || searchParams.price_max) count++
    if (searchParams.rating) count++
    return count
  }

  const activeFiltersCount = getActiveFiltersCount()

  return (
    <div className="bg-white shadow-sm border-b">
      <div className="container-custom py-6">
        {/* Search Form */}
        <form onSubmit={handleSearch} className="mb-6">
          <div className="flex flex-col md:flex-row gap-4">
            <div className="flex-1 relative">
              <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
              <input
                type="text"
                placeholder="Dans türü, kurs adı veya öğretmen ara..."
                value={searchQuery}
                onChange={(e) => setSearchQuery(e.target.value)}
                className="w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent"
              />
            </div>
            <button
              type="submit"
              className="btn-primary px-8 py-3"
            >
              Ara
            </button>
          </div>
        </form>

        {/* Active Filters & Results Info */}
        <div className="flex flex-col md:flex-row md:items-center md:justify-between gap-4">
          <div className="flex items-center space-x-4">
            {/* Active Filters */}
            {activeFiltersCount > 0 && (
              <div className="flex items-center space-x-2">
                <span className="text-sm text-gray-600">Aktif filtreler:</span>
                <div className="flex flex-wrap gap-2">
                  {searchParams.city && (
                    <span className="bg-primary-100 text-primary-800 px-2 py-1 rounded-full text-xs">
                      Şehir: {searchParams.city}
                    </span>
                  )}
                  {searchParams.type && (
                    <span className="bg-primary-100 text-primary-800 px-2 py-1 rounded-full text-xs">
                      Tür: {searchParams.type}
                    </span>
                  )}
                  {searchParams.level && (
                    <span className="bg-primary-100 text-primary-800 px-2 py-1 rounded-full text-xs">
                      Seviye: {searchParams.level}
                    </span>
                  )}
                  {(searchParams.price_min || searchParams.price_max) && (
                    <span className="bg-primary-100 text-primary-800 px-2 py-1 rounded-full text-xs">
                      Fiyat: {searchParams.price_min || '0'}-{searchParams.price_max || '∞'} TL
                    </span>
                  )}
                  {searchParams.rating && (
                    <span className="bg-primary-100 text-primary-800 px-2 py-1 rounded-full text-xs">
                      Puan: {searchParams.rating}+
                    </span>
                  )}
                </div>
                <button
                  onClick={clearFilters}
                  className="text-primary-600 hover:text-primary-700 text-sm flex items-center space-x-1"
                >
                  <X className="w-4 h-4" />
                  <span>Temizle</span>
                </button>
              </div>
            )}

            {/* Results Count */}
            {searchParams.q && (
              <div className="text-sm text-gray-600">
                "{searchParams.q}" için sonuçlar
              </div>
            )}
          </div>

          {/* Mobile Filter Button */}
          <button
            onClick={() => setShowMobileFilters(!showMobileFilters)}
            className="md:hidden flex items-center space-x-2 px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50"
          >
            <Filter className="w-4 h-4" />
            <span>Filtreler</span>
            {activeFiltersCount > 0 && (
              <span className="bg-primary-600 text-white text-xs rounded-full w-5 h-5 flex items-center justify-center">
                {activeFiltersCount}
              </span>
            )}
          </button>
        </div>

        {/* Mobile Filters */}
        {showMobileFilters && (
          <div className="md:hidden mt-4 p-4 bg-gray-50 rounded-lg">
            <SearchFilters searchParams={searchParams} />
          </div>
        )}
      </div>
    </div>
  )
}

