'use client'

import { useRouter, useSearchParams } from 'next/navigation'
import { ChevronLeft, ChevronRight } from 'lucide-react'

interface SearchParams {
  q?: string
  city?: string
  type?: string
  level?: string
  price_min?: string
  price_max?: string
  rating?: string
  page?: string
}

interface SearchPaginationProps {
  searchParams: SearchParams
}

export default function SearchPagination({ searchParams }: SearchPaginationProps) {
  const router = useRouter()
  const urlSearchParams = useSearchParams()
  
  const currentPage = parseInt(searchParams.page || '1')
  const totalPages = 10 // Mock data - gerçek uygulamada API'den gelecek
  const itemsPerPage = 6

  const updatePage = (page: number) => {
    const params = new URLSearchParams(urlSearchParams)
    params.set('page', page.toString())
    router.push(`/arama?${params.toString()}`)
  }

  const getPageNumbers = () => {
    const pages = []
    const maxVisiblePages = 5
    
    if (totalPages <= maxVisiblePages) {
      for (let i = 1; i <= totalPages; i++) {
        pages.push(i)
      }
    } else {
      const startPage = Math.max(1, currentPage - 2)
      const endPage = Math.min(totalPages, startPage + maxVisiblePages - 1)
      
      if (startPage > 1) {
        pages.push(1)
        if (startPage > 2) {
          pages.push('...')
        }
      }
      
      for (let i = startPage; i <= endPage; i++) {
        pages.push(i)
      }
      
      if (endPage < totalPages) {
        if (endPage < totalPages - 1) {
          pages.push('...')
        }
        pages.push(totalPages)
      }
    }
    
    return pages
  }

  if (totalPages <= 1) return null

  return (
    <div className="flex items-center justify-between">
      {/* Results Info */}
      <div className="text-sm text-gray-600">
        Sayfa {currentPage} / {totalPages} 
        <span className="ml-2">
          (Toplam {totalPages * itemsPerPage} kurs)
        </span>
      </div>

      {/* Pagination */}
      <div className="flex items-center space-x-2">
        {/* Previous Button */}
        <button
          onClick={() => updatePage(currentPage - 1)}
          disabled={currentPage === 1}
          className="flex items-center space-x-1 px-3 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
        >
          <ChevronLeft className="w-4 h-4" />
          <span>Önceki</span>
        </button>

        {/* Page Numbers */}
        <div className="flex items-center space-x-1">
          {getPageNumbers().map((page, index) => (
            <button
              key={index}
              onClick={() => typeof page === 'number' && updatePage(page)}
              disabled={page === '...'}
              className={`w-10 h-10 flex items-center justify-center rounded-lg text-sm font-medium transition-colors ${
                page === currentPage
                  ? 'bg-primary-600 text-white'
                  : page === '...'
                  ? 'text-gray-400 cursor-default'
                  : 'text-gray-700 hover:bg-gray-100'
              }`}
            >
              {page}
            </button>
          ))}
        </div>

        {/* Next Button */}
        <button
          onClick={() => updatePage(currentPage + 1)}
          disabled={currentPage === totalPages}
          className="flex items-center space-x-1 px-3 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
        >
          <span>Sonraki</span>
          <ChevronRight className="w-4 h-4" />
        </button>
      </div>
    </div>
  )
}

