'use client'

import { useState, useEffect } from 'react'
import Link from 'next/link'
import Image from 'next/image'
import { Star, MapPin, Clock, Users, Heart, Phone, MessageCircle } from 'lucide-react'

interface SearchParams {
  q?: string
  city?: string
  type?: string
  level?: string
  price_min?: string
  price_max?: string
  rating?: string
  page?: string
}

interface SearchResultsProps {
  searchParams: SearchParams
}

interface Course {
  id: string
  name: string
  location: string
  danceTypes: string[]
  rating: number
  reviewCount: number
  price: number
  currency: string
  duration: string
  level: string
  schedule: string
  image: string
  description: string
  instructor: string
  phone: string
  features: string[]
}

export default function SearchResults({ searchParams }: SearchResultsProps) {
  const [courses, setCourses] = useState<Course[]>([])
  const [loading, setLoading] = useState(true)
  const [favorites, setFavorites] = useState<number[]>([])

  // Mock data - gerçek uygulamada API'den gelecek
  const mockCourses: Course[] = [
    {
      id: '1',
      name: 'Salsa & Bachata Akademi',
      location: 'Kadıköy, İstanbul',
      danceTypes: ['Salsa', 'Bachata', 'Kizomba'],
      rating: 4.9,
      reviewCount: 127,
      price: 150,
      currency: 'TL',
      duration: '8 hafta',
      level: 'Başlangıç',
      schedule: 'Pazartesi, Çarşamba 19:00-21:00',
      image: 'https://images.unsplash.com/photo-1547036967-23d11aacaee0?w=300&h=200&fit=crop',
      description: 'Profesyonel Latin dans eğitimi. Her seviyeye uygun grup dersleri.',
      instructor: 'Maria Rodriguez',
      phone: '+90 532 123 4567',
      features: ['Profesyonel Öğretmen', 'Küçük Gruplar', 'Sertifika']
    },
    {
      id: '2',
      name: 'Modern Dans Stüdyosu',
      location: 'Çankaya, Ankara',
      danceTypes: ['Modern Dans', 'Contemporary', 'Jazz'],
      rating: 4.8,
      reviewCount: 89,
      price: 120,
      currency: 'TL',
      duration: '10 hafta',
      level: 'Orta',
      schedule: 'Salı, Perşembe 18:30-20:30',
      image: 'https://images.unsplash.com/photo-1518611012118-696072aa579a?w=300&h=200&fit=crop',
      description: 'Yaratıcılığınızı keşfedin. Modern dans teknikleri ve koreografi.',
      instructor: 'Ahmet Yılmaz',
      phone: '+90 312 987 6543',
      features: ['Yaratıcı Koreografi', 'Sahne Deneyimi', 'Grup Performansları']
    },
    {
      id: '3',
      name: 'Tango Passion',
      location: 'Konak, İzmir',
      danceTypes: ['Tango', 'Milonga', 'Vals'],
      rating: 4.9,
      reviewCount: 156,
      price: 180,
      currency: 'TL',
      duration: '12 hafta',
      level: 'Tüm Seviyeler',
      schedule: 'Cumartesi, Pazar 15:00-17:00',
      image: 'https://images.unsplash.com/photo-1508700929628-666bc8bd84ea?w=300&h=200&fit=crop',
      description: 'Tutkulu Arjantin tangosu. Romantik ve zarif dans deneyimi.',
      instructor: 'Carlos & Elena',
      phone: '+90 232 555 7890',
      features: ['Çift Eğitimi', 'Müzik Eğitimi', 'Sosyal Danslar']
    },
    {
      id: '4',
      name: 'Hip-Hop Academy',
      location: 'Beşiktaş, İstanbul',
      danceTypes: ['Hip-Hop', 'Breakdance', 'Popping'],
      rating: 4.7,
      reviewCount: 203,
      price: 100,
      currency: 'TL',
      duration: '6 hafta',
      level: 'Başlangıç',
      schedule: 'Pazartesi, Çarşamba, Cuma 20:00-22:00',
      image: 'https://images.unsplash.com/photo-1518611012118-696072aa579a?w=300&h=200&fit=crop',
      description: 'Şehir kültürünün en enerjik dansı. Genç ve dinamik eğitim.',
      instructor: 'DJ Kaan',
      phone: '+90 532 456 7890',
      features: ['Freestyle', 'Koreografi', 'Müzik Prodüksiyon']
    },
    {
      id: '5',
      name: 'Bale & Dans Okulu',
      location: 'Nilüfer, Bursa',
      danceTypes: ['Klasik Bale', 'Modern Bale', 'Karakter'],
      rating: 4.9,
      reviewCount: 78,
      price: 200,
      currency: 'TL',
      duration: '16 hafta',
      level: 'Tüm Seviyeler',
      schedule: 'Salı, Perşembe, Cumartesi 16:00-18:00',
      image: 'https://images.unsplash.com/photo-1508700929628-666bc8bd84ea?w=300&h=200&fit=crop',
      description: 'Klasik bale eğitimi. Disiplinli ve zarif dans öğrenimi.',
      instructor: 'Natalia Petrov',
      phone: '+90 224 333 4567',
      features: ['Rus Metodu', 'Sahne Sanatları', 'Kostüm Tasarım']
    },
    {
      id: '6',
      name: 'Latin Fire Dance',
      location: 'Muratpaşa, Antalya',
      danceTypes: ['Salsa', 'Bachata', 'Merengue', 'Cha Cha'],
      rating: 4.8,
      reviewCount: 145,
      price: 130,
      currency: 'TL',
      duration: '8 hafta',
      level: 'Başlangıç',
      schedule: 'Pazartesi, Çarşamba, Cuma 19:30-21:30',
      image: 'https://images.unsplash.com/photo-1571019613454-1cb2f99b2d8b?w=300&h=200&fit=crop',
      description: 'Sıcak Latin ritimleri. Eğlenceli ve sosyal dans deneyimi.',
      instructor: 'Isabella Cruz',
      phone: '+90 242 777 8888',
      features: ['Sosyal Danslar', 'Sahne Performansı', 'Müzik Eğitimi']
    }
  ]

  useEffect(() => {
    // Simulate API call
    setLoading(true)
    setTimeout(() => {
      let filteredCourses = [...mockCourses]

      // Apply filters
      if (searchParams.q) {
        const query = searchParams.q.toLowerCase()
        filteredCourses = filteredCourses.filter(course =>
          course.name.toLowerCase().includes(query) ||
          course.description.toLowerCase().includes(query) ||
          course.danceTypes.some(type => type.toLowerCase().includes(query)) ||
          course.instructor.toLowerCase().includes(query)
        )
      }

      if (searchParams.city) {
        filteredCourses = filteredCourses.filter(course =>
          course.location.toLowerCase().includes(searchParams.city!.toLowerCase())
        )
      }

      if (searchParams.type) {
        filteredCourses = filteredCourses.filter(course =>
          course.danceTypes.some(type => 
            type.toLowerCase().includes(searchParams.type!.toLowerCase())
          )
        )
      }

      if (searchParams.level) {
        filteredCourses = filteredCourses.filter(course =>
          course.level === searchParams.level
        )
      }

      if (searchParams.price_min) {
        filteredCourses = filteredCourses.filter(course =>
          course.price >= parseInt(searchParams.price_min!)
        )
      }

      if (searchParams.price_max) {
        filteredCourses = filteredCourses.filter(course =>
          course.price <= parseInt(searchParams.price_max!)
        )
      }

      if (searchParams.rating) {
        filteredCourses = filteredCourses.filter(course =>
          course.rating >= parseFloat(searchParams.rating!)
        )
      }

      setCourses(filteredCourses)
      setLoading(false)
    }, 500)
  }, [searchParams])

  const toggleFavorite = (courseId: string) => {
    const id = parseInt(courseId)
    setFavorites(prev => 
      prev.includes(id) 
        ? prev.filter(favId => favId !== id)
        : [...prev, id]
    )
  }

  if (loading) {
    return (
      <div className="space-y-6">
        {[...Array(6)].map((_, i) => (
          <div key={i} className="bg-white rounded-xl shadow-lg p-6 animate-pulse">
            <div className="flex space-x-4">
              <div className="w-32 h-32 bg-gray-200 rounded-lg"></div>
              <div className="flex-1 space-y-3">
                <div className="h-6 bg-gray-200 rounded w-3/4"></div>
                <div className="h-4 bg-gray-200 rounded w-1/2"></div>
                <div className="h-4 bg-gray-200 rounded w-2/3"></div>
                <div className="flex space-x-2">
                  <div className="h-6 bg-gray-200 rounded w-16"></div>
                  <div className="h-6 bg-gray-200 rounded w-16"></div>
                </div>
              </div>
            </div>
          </div>
        ))}
      </div>
    )
  }

  if (courses.length === 0) {
    return (
      <div className="text-center py-12">
        <div className="w-24 h-24 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
          <span className="text-4xl">🔍</span>
        </div>
        <h3 className="text-xl font-semibold text-gray-900 mb-2">Sonuç bulunamadı</h3>
        <p className="text-gray-600 mb-6">
          Arama kriterlerinize uygun kurs bulunamadı. Filtreleri değiştirmeyi deneyin.
        </p>
        <Link href="/arama" className="btn-primary">
          Filtreleri Temizle
        </Link>
      </div>
    )
  }

  return (
    <div className="space-y-6">
      {/* Results Header */}
      <div className="flex items-center justify-between">
        <h2 className="text-lg font-semibold text-gray-900">
          {courses.length} kurs bulundu
        </h2>
        <div className="text-sm text-gray-600">
          Sayfa {searchParams.page || 1}
        </div>
      </div>

      {/* Courses List */}
      {courses.map((course) => (
        <div key={course.id} className="bg-white rounded-xl shadow-lg p-6 hover:shadow-xl transition-shadow">
          <div className="flex flex-col md:flex-row gap-4">
            {/* Course Image */}
            <div className="relative w-full md:w-48 h-48 md:h-32 flex-shrink-0">
              <Image
                src={course.image}
                alt={course.name}
                fill
                className="object-cover rounded-lg"
              />
              <button
                onClick={() => toggleFavorite(course.id)}
                className="absolute top-2 right-2 p-2 bg-white/90 rounded-full hover:bg-white transition-colors"
              >
                <Heart 
                  className={`w-4 h-4 ${
                    favorites.includes(parseInt(course.id)) 
                      ? 'text-red-500 fill-current' 
                      : 'text-gray-400 hover:text-red-500'
                  }`} 
                />
              </button>
            </div>

            {/* Course Info */}
            <div className="flex-1">
              <div className="flex flex-col md:flex-row md:items-start md:justify-between gap-4">
                <div className="flex-1">
                  <h3 className="text-xl font-bold text-gray-900 mb-2">
                    <Link 
                      href={`/kurslar/${course.id}`}
                      className="hover:text-primary-600 transition-colors"
                    >
                      {course.name}
                    </Link>
                  </h3>
                  
                  <div className="flex items-center text-gray-600 mb-2">
                    <MapPin className="w-4 h-4 mr-1" />
                    <span>{course.location}</span>
                  </div>

                  <p className="text-gray-700 mb-4 line-clamp-2">
                    {course.description}
                  </p>

                  {/* Dance Types */}
                  <div className="flex flex-wrap gap-2 mb-4">
                    {course.danceTypes.map((type) => (
                      <span
                        key={type}
                        className="bg-primary-100 text-primary-800 px-2 py-1 rounded-full text-xs font-medium"
                      >
                        {type}
                      </span>
                    ))}
                  </div>

                  {/* Features */}
                  <div className="flex flex-wrap gap-2">
                    {course.features.slice(0, 3).map((feature) => (
                      <div key={feature} className="flex items-center text-xs text-gray-600">
                        <div className="w-1.5 h-1.5 bg-primary-500 rounded-full mr-2"></div>
                        {feature}
                      </div>
                    ))}
                  </div>
                </div>

                {/* Course Stats & Actions */}
                <div className="flex flex-col items-end space-y-4">
                  {/* Rating & Price */}
                  <div className="text-right">
                    <div className="flex items-center space-x-1 mb-1">
                      <Star className="w-4 h-4 text-yellow-400 fill-current" />
                      <span className="font-semibold">{course.rating}</span>
                      <span className="text-gray-500 text-sm">({course.reviewCount})</span>
                    </div>
                    <div className="text-2xl font-bold text-primary-600">
                      {course.price} {course.currency}
                    </div>
                    <div className="text-sm text-gray-500">aylık</div>
                  </div>

                  {/* Course Details */}
                  <div className="text-right text-sm text-gray-600 space-y-1">
                    <div className="flex items-center space-x-1">
                      <Clock className="w-4 h-4" />
                      <span>{course.duration}</span>
                    </div>
                    <div className="flex items-center space-x-1">
                      <Users className="w-4 h-4" />
                      <span>{course.level}</span>
                    </div>
                  </div>

                  {/* Action Buttons */}
                  <div className="flex space-x-2">
                    <Link
                      href={`/kurslar/${course.id}`}
                      className="btn-primary text-sm px-4 py-2"
                    >
                      Detaylar
                    </Link>
                    <a
                      href={`tel:${course.phone}`}
                      className="p-2 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors"
                    >
                      <Phone className="w-4 h-4" />
                    </a>
                    <a
                      href={`https://wa.me/${course.phone.replace(/[^0-9]/g, '')}`}
                      target="_blank"
                      rel="noopener noreferrer"
                      className="p-2 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors"
                    >
                      <MessageCircle className="w-4 h-4" />
                    </a>
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      ))}
    </div>
  )
}

