<?php
/**
 * Schema.org Structured Data Generator
 *
 * @package FirmaRehberiPro
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Generate company schema
 */
function firma_rehberi_generate_company_schema() {
    if (!is_singular('company')) {
        return false;
    }
    
    global $post;
    
    $company_name = get_the_title();
    $description = get_the_excerpt() ?: get_the_content();
    $phone = get_post_meta($post->ID, '_phone', true);
    $email = get_post_meta($post->ID, '_email', true);
    $website = get_post_meta($post->ID, '_website', true);
    $address = get_post_meta($post->ID, '_address', true);
    $latitude = get_post_meta($post->ID, '_latitude', true);
    $longitude = get_post_meta($post->ID, '_longitude', true);
    $working_hours = get_post_meta($post->ID, '_working_hours', true);
    $rating = get_post_meta($post->ID, '_rating', true);
    $review_count = get_post_meta($post->ID, '_review_count', true) ?: 0;
    
    // Get categories
    $categories = get_the_terms($post->ID, 'company_category');
    $category_name = $categories && !is_wp_error($categories) ? $categories[0]->name : '';
    
    // Get cities
    $cities = get_the_terms($post->ID, 'company_city');
    $city_name = $cities && !is_wp_error($cities) ? $cities[0]->name : '';
    
    // Get social media
    $facebook = get_post_meta($post->ID, '_facebook', true);
    $twitter = get_post_meta($post->ID, '_twitter', true);
    $instagram = get_post_meta($post->ID, '_instagram', true);
    $linkedin = get_post_meta($post->ID, '_linkedin', true);
    
    $schema = array(
        '@context' => 'https://schema.org',
        '@type' => array('LocalBusiness', 'Organization'),
        'name' => $company_name,
        'description' => wp_trim_words($description, 30),
        'url' => get_permalink(),
        'image' => get_the_post_thumbnail_url($post->ID, 'large'),
    );
    
    // Add contact information
    if ($phone) {
        $schema['telephone'] = $phone;
    }
    
    if ($email) {
        $schema['email'] = $email;
    }
    
    if ($website) {
        $schema['url'] = $website;
    }
    
    // Add address
    if ($address) {
        $schema['address'] = array(
            '@type' => 'PostalAddress',
            'streetAddress' => $address,
            'addressLocality' => $city_name,
            'addressCountry' => 'TR'
        );
    }
    
    // Add geo coordinates
    if ($latitude && $longitude) {
        $schema['geo'] = array(
            '@type' => 'GeoCoordinates',
            'latitude' => floatval($latitude),
            'longitude' => floatval($longitude)
        );
    }
    
    // Add opening hours
    if ($working_hours) {
        $schema['openingHours'] = $working_hours;
    }
    
    // Add category
    if ($category_name) {
        $schema['category'] = $category_name;
    }
    
    // Add social media
    $same_as = array();
    if ($facebook) $same_as[] = $facebook;
    if ($twitter) $same_as[] = $twitter;
    if ($instagram) $same_as[] = $instagram;
    if ($linkedin) $same_as[] = $linkedin;
    
    if (!empty($same_as)) {
        $schema['sameAs'] = $same_as;
    }
    
    // Add rating
    if ($rating && $review_count > 0) {
        $schema['aggregateRating'] = array(
            '@type' => 'AggregateRating',
            'ratingValue' => floatval($rating),
            'reviewCount' => intval($review_count),
            'bestRating' => 5,
            'worstRating' => 1
        );
    }
    
    // Add reviews
    $reviews = get_comments(array(
        'post_id' => $post->ID,
        'status' => 'approve',
        'number' => 5,
    ));
    
    if ($reviews) {
        $schema['review'] = array();
        foreach ($reviews as $review) {
            $review_rating = get_comment_meta($review->comment_ID, '_rating', true);
            if ($review_rating) {
                $schema['review'][] = array(
                    '@type' => 'Review',
                    'author' => array(
                        '@type' => 'Person',
                        'name' => $review->comment_author
                    ),
                    'reviewRating' => array(
                        '@type' => 'Rating',
                        'ratingValue' => intval($review_rating),
                        'bestRating' => 5,
                        'worstRating' => 1
                    ),
                    'reviewBody' => $review->comment_content,
                    'datePublished' => get_comment_date('c', $review)
                );
            }
        }
    }
    
    return $schema;
}

/**
 * Generate archive schema
 */
function firma_rehberi_generate_archive_schema() {
    if (!is_post_type_archive('company') && !is_tax('company_category') && !is_tax('company_city')) {
        return false;
    }
    
    $schema = array(
        '@context' => 'https://schema.org',
        '@type' => 'ItemList',
        'name' => '',
        'description' => '',
        'url' => '',
        'numberOfItems' => 0,
        'itemListElement' => array()
    );
    
    if (is_post_type_archive('company')) {
        $schema['name'] = __('Firmalar', 'firma-rehberi-pro');
        $schema['description'] = __('Türkiye\'nin en kapsamlı firma rehberi', 'firma-rehberi-pro');
        $schema['url'] = get_post_type_archive_link('company');
    } elseif (is_tax('company_category')) {
        $term = get_queried_object();
        $schema['name'] = $term->name . ' ' . __('Firmaları', 'firma-rehberi-pro');
        $schema['description'] = sprintf(__('%s kategorisindeki firmalar', 'firma-rehberi-pro'), $term->name);
        $schema['url'] = get_term_link($term);
    } elseif (is_tax('company_city')) {
        $term = get_queried_object();
        $schema['name'] = $term->name . ' ' . __('Firmaları', 'firma-rehberi-pro');
        $schema['description'] = sprintf(__('%s şehrindeki firmalar', 'firma-rehberi-pro'), $term->name);
        $schema['url'] = get_term_link($term);
    }
    
    global $wp_query;
    $schema['numberOfItems'] = $wp_query->found_posts;
    
    // Add companies to list
    if (have_posts()) {
        $position = 1;
        while (have_posts()) {
            the_post();
            
            $company_schema = array(
                '@type' => 'ListItem',
                'position' => $position,
                'item' => array(
                    '@type' => 'LocalBusiness',
                    'name' => get_the_title(),
                    'url' => get_permalink(),
                    'image' => get_the_post_thumbnail_url(get_the_ID(), 'medium'),
                    'description' => get_the_excerpt()
                )
            );
            
            // Add address if available
            $address = get_post_meta(get_the_ID(), '_address', true);
            $cities = get_the_terms(get_the_ID(), 'company_city');
            if ($address && $cities && !is_wp_error($cities)) {
                $company_schema['item']['address'] = array(
                    '@type' => 'PostalAddress',
                    'streetAddress' => $address,
                    'addressLocality' => $cities[0]->name,
                    'addressCountry' => 'TR'
                );
            }
            
            $schema['itemListElement'][] = $company_schema;
            $position++;
        }
        wp_reset_postdata();
    }
    
    return $schema;
}

/**
 * Generate website schema
 */
function firma_rehberi_generate_website_schema() {
    if (!is_home() && !is_front_page()) {
        return false;
    }
    
    $schema = array(
        '@context' => 'https://schema.org',
        '@type' => 'WebSite',
        'name' => get_bloginfo('name'),
        'description' => get_bloginfo('description'),
        'url' => home_url('/'),
        'potentialAction' => array(
            '@type' => 'SearchAction',
            'target' => home_url('/?s={search_term_string}'),
            'query-input' => 'required name=search_term_string'
        )
    );
    
    // Add organization info
    $schema['publisher'] = array(
        '@type' => 'Organization',
        'name' => get_bloginfo('name'),
        'url' => home_url('/'),
        'logo' => array(
            '@type' => 'ImageObject',
            'url' => get_theme_mod('custom_logo') ? wp_get_attachment_url(get_theme_mod('custom_logo')) : '',
            'width' => 300,
            'height' => 100
        )
    );
    
    return $schema;
}

/**
 * Generate breadcrumb schema
 */
function firma_rehberi_generate_breadcrumb_schema() {
    if (is_front_page()) {
        return false;
    }
    
    $breadcrumbs = array();
    $breadcrumbs[] = array(
        'name' => __('Ana Sayfa', 'firma-rehberi-pro'),
        'url'  => home_url('/'),
    );
    
    if (is_singular('company')) {
        $breadcrumbs[] = array(
            'name' => __('Firmalar', 'firma-rehberi-pro'),
            'url'  => get_post_type_archive_link('company'),
        );
        
        $categories = get_the_terms(get_the_ID(), 'company_category');
        if ($categories && !is_wp_error($categories)) {
            $breadcrumbs[] = array(
                'name' => $categories[0]->name,
                'url'  => get_term_link($categories[0]),
            );
        }
        
        $breadcrumbs[] = array(
            'name' => get_the_title(),
            'url'  => get_permalink(),
        );
    } elseif (is_post_type_archive('company')) {
        $breadcrumbs[] = array(
            'name' => __('Firmalar', 'firma-rehberi-pro'),
            'url'  => get_post_type_archive_link('company'),
        );
    } elseif (is_tax('company_category')) {
        $breadcrumbs[] = array(
            'name' => __('Firmalar', 'firma-rehberi-pro'),
            'url'  => get_post_type_archive_link('company'),
        );
        
        $term = get_queried_object();
        $breadcrumbs[] = array(
            'name' => $term->name,
            'url'  => get_term_link($term),
        );
    } elseif (is_tax('company_city')) {
        $breadcrumbs[] = array(
            'name' => __('Firmalar', 'firma-rehberi-pro'),
            'url'  => get_post_type_archive_link('company'),
        );
        
        $term = get_queried_object();
        $breadcrumbs[] = array(
            'name' => $term->name,
            'url'  => get_term_link($term),
        );
    } elseif (is_singular()) {
        $breadcrumbs[] = array(
            'name' => get_the_title(),
            'url'  => get_permalink(),
        );
    } elseif (is_category()) {
        $breadcrumbs[] = array(
            'name' => __('Haberler', 'firma-rehberi-pro'),
            'url'  => get_permalink(get_option('page_for_posts')),
        );
        
        $term = get_queried_object();
        $breadcrumbs[] = array(
            'name' => $term->name,
            'url'  => get_term_link($term),
        );
    } elseif (is_search()) {
        $breadcrumbs[] = array(
            'name' => sprintf(__('Arama sonuçları: "%s"', 'firma-rehberi-pro'), get_search_query()),
            'url'  => get_search_link(),
        );
    }
    
    $schema = array(
        '@context' => 'https://schema.org',
        '@type' => 'BreadcrumbList',
        'itemListElement' => array()
    );
    
    foreach ($breadcrumbs as $index => $breadcrumb) {
        $schema['itemListElement'][] = array(
            '@type' => 'ListItem',
            'position' => $index + 1,
            'name' => $breadcrumb['name'],
            'item' => $breadcrumb['url']
        );
    }
    
    return $schema;
}

/**
 * Generate blog post schema
 */
function firma_rehberi_generate_blog_post_schema() {
    if (!is_singular('post')) {
        return false;
    }
    
    global $post;
    
    $schema = array(
        '@context' => 'https://schema.org',
        '@type' => 'BlogPosting',
        'headline' => get_the_title(),
        'description' => get_the_excerpt(),
        'url' => get_permalink(),
        'datePublished' => get_the_date('c'),
        'dateModified' => get_the_modified_date('c'),
        'author' => array(
            '@type' => 'Person',
            'name' => get_the_author(),
            'url' => get_author_posts_url(get_the_author_meta('ID'))
        ),
        'publisher' => array(
            '@type' => 'Organization',
            'name' => get_bloginfo('name'),
            'url' => home_url('/'),
            'logo' => array(
                '@type' => 'ImageObject',
                'url' => get_theme_mod('custom_logo') ? wp_get_attachment_url(get_theme_mod('custom_logo')) : '',
                'width' => 300,
                'height' => 100
            )
        ),
        'mainEntityOfPage' => array(
            '@type' => 'WebPage',
            '@id' => get_permalink()
        )
    );
    
    // Add featured image
    if (has_post_thumbnail()) {
        $schema['image'] = array(
            '@type' => 'ImageObject',
            'url' => get_the_post_thumbnail_url($post->ID, 'large'),
            'width' => 1200,
            'height' => 630
        );
    }
    
    // Add categories
    $categories = get_the_category();
    if ($categories && !is_wp_error($categories)) {
        $schema['articleSection'] = $categories[0]->name;
        $schema['keywords'] = implode(', ', wp_list_pluck($categories, 'name'));
    }
    
    // Add word count
    $word_count = str_word_count(strip_tags(get_the_content()));
    if ($word_count > 0) {
        $schema['wordCount'] = $word_count;
    }
    
    return $schema;
}

/**
 * Add schema to head
 */
function firma_rehberi_add_schema_to_head() {
    $schema = false;
    
    if (is_singular('company')) {
        $schema = firma_rehberi_generate_company_schema();
    } elseif (is_post_type_archive('company') || is_tax('company_category') || is_tax('company_city')) {
        $schema = firma_rehberi_generate_archive_schema();
    } elseif (is_home() || is_front_page()) {
        $schema = firma_rehberi_generate_website_schema();
    } elseif (is_singular('post')) {
        $schema = firma_rehberi_generate_blog_post_schema();
    }
    
    if ($schema) {
        echo '<script type="application/ld+json">' . wp_json_encode($schema, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE) . '</script>' . "\n";
    }
}
add_action('wp_head', 'firma_rehberi_add_schema_to_head');
