// SEO Utility Functions

export interface PageSEO {
  title: string
  description: string
  keywords?: string[]
  canonical?: string
  ogImage?: string
  noindex?: boolean
  nofollow?: boolean
}

/**
 * Generate optimized page title
 * Max length: 60 characters for optimal display
 */
export function generatePageTitle(title: string, includeSiteName: boolean = true): string {
  const siteName = 'Dans Dersleri'
  const separator = ' | '
  
  if (!includeSiteName) {
    return title.slice(0, 60)
  }
  
  const fullTitle = `${title}${separator}${siteName}`
  
  if (fullTitle.length <= 60) {
    return fullTitle
  }
  
  // Truncate title to fit with site name
  const maxTitleLength = 60 - separator.length - siteName.length - 3 // 3 for "..."
  return `${title.slice(0, maxTitleLength)}...${separator}${siteName}`
}

/**
 * Generate optimized meta description
 * Max length: 160 characters for optimal display
 */
export function generateMetaDescription(description: string): string {
  if (description.length <= 160) {
    return description
  }
  
  return description.slice(0, 157) + '...'
}

/**
 * Generate keywords string from array
 */
export function generateKeywords(keywords: string[]): string {
  return keywords.join(', ')
}

/**
 * Generate canonical URL
 */
export function generateCanonicalUrl(path: string): string {
  const baseUrl = process.env.NEXT_PUBLIC_BASE_URL || 'https://dansdersleri.com'
  const cleanPath = path.startsWith('/') ? path : `/${path}`
  return `${baseUrl}${cleanPath}`
}

/**
 * Generate OG Image URL
 */
export function generateOgImageUrl(image?: string): string {
  if (image) {
    if (image.startsWith('http')) {
      return image
    }
    const baseUrl = process.env.NEXT_PUBLIC_BASE_URL || 'https://dansdersleri.com'
    return `${baseUrl}${image}`
  }
  
  return 'https://dansdersleri.com/og-image.jpg'
}

/**
 * Generate breadcrumb items from path
 */
export function generateBreadcrumbs(path: string, customNames?: Record<string, string>) {
  const baseUrl = process.env.NEXT_PUBLIC_BASE_URL || 'https://dansdersleri.com'
  const pathSegments = path.split('/').filter(Boolean)
  
  const breadcrumbs = [
    { name: 'Ana Sayfa', url: baseUrl }
  ]
  
  let currentPath = ''
  pathSegments.forEach((segment, index) => {
    currentPath += `/${segment}`
    const name = customNames?.[segment] || capitalizeWords(segment.replace(/-/g, ' '))
    breadcrumbs.push({
      name,
      url: `${baseUrl}${currentPath}`
    })
  })
  
  return breadcrumbs
}

/**
 * Capitalize words in a string
 */
function capitalizeWords(str: string): string {
  return str
    .split(' ')
    .map(word => word.charAt(0).toUpperCase() + word.slice(1).toLowerCase())
    .join(' ')
}

/**
 * Generate alt text for images
 */
export function generateImageAlt(context: string, description?: string): string {
  if (description) {
    return `${context} - ${description}`
  }
  return context
}

/**
 * Clean and optimize URL slug
 */
export function generateSlug(text: string): string {
  return text
    .toLowerCase()
    .trim()
    .replace(/[\s_]+/g, '-')  // Replace spaces and underscores with hyphens
    .replace(/[^a-z0-9-]/g, '') // Remove non-alphanumeric characters except hyphens
    .replace(/-+/g, '-')  // Replace multiple hyphens with single hyphen
    .replace(/^-|-$/g, '') // Remove leading/trailing hyphens
}

/**
 * Extract keywords from text
 */
export function extractKeywords(text: string, count: number = 10): string[] {
  // Common Turkish stop words
  const stopWords = [
    've', 'ile', 'için', 'bir', 'bu', 'şu', 'o', 'da', 'de', 
    'mi', 'mı', 'mu', 'mü', 'gibi', 'ancak', 'ama', 'fakat'
  ]
  
  const words = text
    .toLowerCase()
    .replace(/[^a-zçğıöşü\s]/gi, '')
    .split(/\s+/)
    .filter(word => word.length > 3 && !stopWords.includes(word))
  
  // Count word frequency
  const wordCount: Record<string, number> = {}
  words.forEach(word => {
    wordCount[word] = (wordCount[word] || 0) + 1
  })
  
  // Sort by frequency and return top keywords
  return Object.entries(wordCount)
    .sort((a, b) => b[1] - a[1])
    .slice(0, count)
    .map(([word]) => word)
}

/**
 * Generate reading time estimate
 */
export function calculateReadingTime(text: string): number {
  const wordsPerMinute = 200
  const wordCount = text.split(/\s+/).length
  return Math.ceil(wordCount / wordsPerMinute)
}

/**
 * Check if URL is absolute
 */
export function isAbsoluteUrl(url: string): boolean {
  return /^https?:\/\//i.test(url)
}

/**
 * Generate social share URLs
 */
export function generateShareUrls(url: string, title: string) {
  const encodedUrl = encodeURIComponent(url)
  const encodedTitle = encodeURIComponent(title)
  
  return {
    facebook: `https://www.facebook.com/sharer/sharer.php?u=${encodedUrl}`,
    twitter: `https://twitter.com/intent/tweet?url=${encodedUrl}&text=${encodedTitle}`,
    linkedin: `https://www.linkedin.com/shareArticle?mini=true&url=${encodedUrl}&title=${encodedTitle}`,
    whatsapp: `https://wa.me/?text=${encodedTitle}%20${encodedUrl}`,
    telegram: `https://t.me/share/url?url=${encodedUrl}&text=${encodedTitle}`,
    email: `mailto:?subject=${encodedTitle}&body=${encodedUrl}`
  }
}

/**
 * Validate and sanitize URL
 */
export function sanitizeUrl(url: string): string {
  try {
    const urlObj = new URL(url)
    return urlObj.href
  } catch {
    return '/'
  }
}

