<?php
/**
 * Helper Functions
 *
 * @package FirmaRehberiPro
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Get company rating stars HTML
 */
function firma_rehberi_get_rating_stars($rating, $size = 'normal') {
    $rating = floatval($rating);
    $stars = '';
    
    $size_class = $size === 'large' ? 'text-xl' : 'text-base';
    
    for ($i = 1; $i <= 5; $i++) {
        if ($i <= $rating) {
            $stars .= '<span class="star filled ' . $size_class . '">★</span>';
        } elseif ($i - 0.5 <= $rating) {
            $stars .= '<span class="star half-filled ' . $size_class . '">☆</span>';
        } else {
            $stars .= '<span class="star empty ' . $size_class . '">☆</span>';
        }
    }
    
    return $stars;
}

/**
 * Get company view count
 */
function firma_rehberi_get_view_count($post_id = null) {
    if (!$post_id) {
        $post_id = get_the_ID();
    }
    
    $views = get_post_meta($post_id, '_view_count', true);
    return $views ? intval($views) : 0;
}

/**
 * Format view count with K/M suffixes
 */
function firma_rehberi_format_view_count($count) {
    if ($count >= 1000000) {
        return round($count / 1000000, 1) . 'M';
    } elseif ($count >= 1000) {
        return round($count / 1000, 1) . 'K';
    }
    
    return $count;
}

/**
 * Get company categories as links
 */
function firma_rehberi_get_company_categories($post_id = null, $linked = true) {
    if (!$post_id) {
        $post_id = get_the_ID();
    }
    
    $categories = get_the_terms($post_id, 'company_category');
    
    if (!$categories || is_wp_error($categories)) {
        return '';
    }
    
    $output = array();
    
    foreach ($categories as $category) {
        if ($linked) {
            $output[] = '<a href="' . esc_url(get_term_link($category)) . '" class="category-link">' . esc_html($category->name) . '</a>';
        } else {
            $output[] = esc_html($category->name);
        }
    }
    
    return implode(', ', $output);
}

/**
 * Get company cities as links
 */
function firma_rehberi_get_company_cities($post_id = null, $linked = true) {
    if (!$post_id) {
        $post_id = get_the_ID();
    }
    
    $cities = get_the_terms($post_id, 'company_city');
    
    if (!$cities || is_wp_error($cities)) {
        return '';
    }
    
    $output = array();
    
    foreach ($cities as $city) {
        if ($linked) {
            $output[] = '<a href="' . esc_url(get_term_link($city)) . '" class="city-link">' . esc_html($city->name) . '</a>';
        } else {
            $output[] = esc_html($city->name);
        }
    }
    
    return implode(', ', $output);
}

/**
 * Get company contact info
 */
function firma_rehberi_get_company_contact_info($post_id = null) {
    if (!$post_id) {
        $post_id = get_the_ID();
    }
    
    $contact_info = array();
    
    $phone = get_post_meta($post_id, '_phone', true);
    if ($phone) {
        $contact_info['phone'] = $phone;
    }
    
    $email = get_post_meta($post_id, '_email', true);
    if ($email) {
        $contact_info['email'] = $email;
    }
    
    $website = get_post_meta($post_id, '_website', true);
    if ($website) {
        $contact_info['website'] = $website;
    }
    
    $address = get_post_meta($post_id, '_address', true);
    if ($address) {
        $contact_info['address'] = $address;
    }
    
    return $contact_info;
}

/**
 * Get company social media links
 */
function firma_rehberi_get_company_social_links($post_id = null) {
    if (!$post_id) {
        $post_id = get_the_ID();
    }
    
    $social_links = array();
    
    $facebook = get_post_meta($post_id, '_facebook', true);
    if ($facebook) {
        $social_links['facebook'] = $facebook;
    }
    
    $twitter = get_post_meta($post_id, '_twitter', true);
    if ($twitter) {
        $social_links['twitter'] = $twitter;
    }
    
    $instagram = get_post_meta($post_id, '_instagram', true);
    if ($instagram) {
        $social_links['instagram'] = $instagram;
    }
    
    $linkedin = get_post_meta($post_id, '_linkedin', true);
    if ($linkedin) {
        $social_links['linkedin'] = $linkedin;
    }
    
    return $social_links;
}

/**
 * Check if company is featured
 */
function firma_rehberi_is_featured_company($post_id = null) {
    if (!$post_id) {
        $post_id = get_the_ID();
    }
    
    return get_post_meta($post_id, '_featured', true) === '1';
}

/**
 * Get company services as array
 */
function firma_rehberi_get_company_services($post_id = null) {
    if (!$post_id) {
        $post_id = get_the_ID();
    }
    
    $services = get_post_meta($post_id, '_services', true);
    
    if (!$services) {
        return array();
    }
    
    return array_filter(array_map('trim', explode("\n", $services)));
}

/**
 * Get company gallery images
 */
function firma_rehberi_get_company_gallery($post_id = null) {
    if (!$post_id) {
        $post_id = get_the_ID();
    }
    
    $gallery = get_post_meta($post_id, '_gallery', true);
    
    if (!$gallery) {
        return array();
    }
    
    $gallery_ids = array_filter(explode(',', $gallery));
    $images = array();
    
    foreach ($gallery_ids as $image_id) {
        $image = wp_get_attachment_image_src($image_id, 'large');
        if ($image) {
            $images[] = array(
                'id' => $image_id,
                'url' => $image[0],
                'width' => $image[1],
                'height' => $image[2],
                'alt' => get_post_meta($image_id, '_wp_attachment_image_alt', true),
            );
        }
    }
    
    return $images;
}

/**
 * Get company coordinates
 */
function firma_rehberi_get_company_coordinates($post_id = null) {
    if (!$post_id) {
        $post_id = get_the_ID();
    }
    
    $latitude = get_post_meta($post_id, '_latitude', true);
    $longitude = get_post_meta($post_id, '_longitude', true);
    
    if ($latitude && $longitude) {
        return array(
            'latitude' => floatval($latitude),
            'longitude' => floatval($longitude),
        );
    }
    
    return false;
}

/**
 * Get company working hours
 */
function firma_rehberi_get_company_working_hours($post_id = null) {
    if (!$post_id) {
        $post_id = get_the_ID();
    }
    
    return get_post_meta($post_id, '_working_hours', true);
}

/**
 * Get company rating and review count
 */
function firma_rehberi_get_company_rating_info($post_id = null) {
    if (!$post_id) {
        $post_id = get_the_ID();
    }
    
    $rating = get_post_meta($post_id, '_rating', true);
    $review_count = get_post_meta($post_id, '_review_count', true);
    
    return array(
        'rating' => $rating ? floatval($rating) : 0,
        'review_count' => $review_count ? intval($review_count) : 0,
    );
}

/**
 * Get popular companies
 */
function firma_rehberi_get_popular_companies($limit = 5) {
    $companies = get_posts(array(
        'post_type' => 'company',
        'posts_per_page' => $limit,
        'meta_key' => '_view_count',
        'orderby' => 'meta_value_num',
        'order' => 'DESC',
        'post_status' => 'publish',
    ));
    
    return $companies;
}

/**
 * Get featured companies
 */
function firma_rehberi_get_featured_companies($limit = 5) {
    $companies = get_posts(array(
        'post_type' => 'company',
        'posts_per_page' => $limit,
        'meta_key' => '_featured',
        'meta_value' => '1',
        'orderby' => 'rand',
        'post_status' => 'publish',
    ));
    
    return $companies;
}

/**
 * Get recent companies
 */
function firma_rehberi_get_recent_companies($limit = 5) {
    $companies = get_posts(array(
        'post_type' => 'company',
        'posts_per_page' => $limit,
        'orderby' => 'date',
        'order' => 'DESC',
        'post_status' => 'publish',
    ));
    
    return $companies;
}

/**
 * Get companies by category
 */
function firma_rehberi_get_companies_by_category($category_slug, $limit = 10) {
    $companies = get_posts(array(
        'post_type' => 'company',
        'posts_per_page' => $limit,
        'tax_query' => array(
            array(
                'taxonomy' => 'company_category',
                'field' => 'slug',
                'terms' => $category_slug,
            ),
        ),
        'orderby' => 'date',
        'order' => 'DESC',
        'post_status' => 'publish',
    ));
    
    return $companies;
}

/**
 * Get companies by city
 */
function firma_rehberi_get_companies_by_city($city_slug, $limit = 10) {
    $companies = get_posts(array(
        'post_type' => 'company',
        'posts_per_page' => $limit,
        'tax_query' => array(
            array(
                'taxonomy' => 'company_city',
                'field' => 'slug',
                'terms' => $city_slug,
            ),
        ),
        'orderby' => 'date',
        'order' => 'DESC',
        'post_status' => 'publish',
    ));
    
    return $companies;
}

/**
 * Get company statistics
 */
function firma_rehberi_get_company_stats() {
    $stats = array();
    
    // Total companies
    $stats['total_companies'] = wp_count_posts('company')->publish;
    
    // Total categories
    $stats['total_categories'] = wp_count_terms('company_category');
    
    // Total cities
    $stats['total_cities'] = wp_count_terms('company_city');
    
    // Featured companies
    $featured_query = new WP_Query(array(
        'post_type' => 'company',
        'meta_key' => '_featured',
        'meta_value' => '1',
        'posts_per_page' => -1,
        'fields' => 'ids',
    ));
    $stats['featured_companies'] = $featured_query->found_posts;
    
    // Total views
    global $wpdb;
    $stats['total_views'] = $wpdb->get_var("SELECT SUM(meta_value) FROM {$wpdb->postmeta} WHERE meta_key = '_view_count'");
    
    return $stats;
}

/**
 * Sanitize phone number
 */
function firma_rehberi_sanitize_phone($phone) {
    // Remove all non-numeric characters except + and spaces
    $phone = preg_replace('/[^0-9+\s]/', '', $phone);
    
    // Remove extra spaces
    $phone = preg_replace('/\s+/', ' ', trim($phone));
    
    return $phone;
}

/**
 * Format phone number for display
 */
function firma_rehberi_format_phone($phone) {
    $phone = firma_rehberi_sanitize_phone($phone);
    
    // Add spaces for better readability
    if (strlen($phone) >= 10) {
        $phone = preg_replace('/(\d{3})(\d{3})(\d{4})/', '$1 $2 $3', $phone);
    }
    
    return $phone;
}

/**
 * Get company search suggestions
 */
function firma_rehberi_get_search_suggestions($query, $limit = 5) {
    $suggestions = array();
    
    // Search companies
    $companies = get_posts(array(
        'post_type' => 'company',
        'posts_per_page' => $limit,
        's' => $query,
        'post_status' => 'publish',
    ));
    
    foreach ($companies as $company) {
        $suggestions[] = array(
            'type' => 'company',
            'title' => $company->post_title,
            'url' => get_permalink($company->ID),
            'description' => get_the_excerpt($company->ID),
        );
    }
    
    // Search categories
    $categories = get_terms(array(
        'taxonomy' => 'company_category',
        'name__like' => $query,
        'number' => $limit,
        'hide_empty' => true,
    ));
    
    foreach ($categories as $category) {
        $suggestions[] = array(
            'type' => 'category',
            'title' => $category->name,
            'url' => get_term_link($category),
            'description' => $category->description,
        );
    }
    
    return $suggestions;
}

/**
 * Check if current page is company related
 */
function firma_rehberi_is_company_page() {
    return is_singular('company') || is_post_type_archive('company') || is_tax('company_category') || is_tax('company_city');
}

/**
 * Get breadcrumb items
 */
function firma_rehberi_get_breadcrumb_items() {
    $items = array();
    
    // Always start with home
    $items[] = array(
        'name' => __('Ana Sayfa', 'firma-rehberi-pro'),
        'url' => home_url('/'),
    );
    
    if (is_singular('company')) {
        $items[] = array(
            'name' => __('Firmalar', 'firma-rehberi-pro'),
            'url' => get_post_type_archive_link('company'),
        );
        
        $categories = get_the_terms(get_the_ID(), 'company_category');
        if ($categories && !is_wp_error($categories)) {
            $items[] = array(
                'name' => $categories[0]->name,
                'url' => get_term_link($categories[0]),
            );
        }
        
        $items[] = array(
            'name' => get_the_title(),
            'url' => '',
        );
    } elseif (is_post_type_archive('company')) {
        $items[] = array(
            'name' => __('Firmalar', 'firma-rehberi-pro'),
            'url' => '',
        );
    } elseif (is_tax('company_category')) {
        $items[] = array(
            'name' => __('Firmalar', 'firma-rehberi-pro'),
            'url' => get_post_type_archive_link('company'),
        );
        
        $term = get_queried_object();
        $items[] = array(
            'name' => $term->name,
            'url' => '',
        );
    } elseif (is_tax('company_city')) {
        $items[] = array(
            'name' => __('Firmalar', 'firma-rehberi-pro'),
            'url' => get_post_type_archive_link('company'),
        );
        
        $term = get_queried_object();
        $items[] = array(
            'name' => $term->name,
            'url' => '',
        );
    }
    
    return $items;
}
