<?php
/**
 * PHP 8.1 Compatibility Functions
 *
 * @package FirmaRehberiPro
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Check PHP version compatibility
 */
function firma_rehberi_check_php_version() {
    if (version_compare(PHP_VERSION, '8.1.0', '<')) {
        add_action('admin_notices', 'firma_rehberi_php_version_notice');
        return false;
    }
    return true;
}
add_action('admin_init', 'firma_rehberi_check_php_version');

/**
 * PHP version notice
 */
function firma_rehberi_php_version_notice() {
    $message = sprintf(
        __('Firma Rehberi Pro teması PHP 8.1 veya üzeri gerektirir. Şu anki PHP sürümünüz: %s', 'firma-rehberi-pro'),
        PHP_VERSION
    );
    echo '<div class="notice notice-error"><p>' . esc_html($message) . '</p></div>';
}

/**
 * PHP 8.1 compatible array functions
 */
if (!function_exists('array_key_first')) {
    function array_key_first($array) {
        if (!is_array($array) || empty($array)) {
            return null;
        }
        return key($array);
    }
}

if (!function_exists('array_key_last')) {
    function array_key_last($array) {
        if (!is_array($array) || empty($array)) {
            return null;
        }
        end($array);
        return key($array);
    }
}

/**
 * PHP 8.1 compatible string functions
 */
if (!function_exists('str_contains')) {
    function str_contains($haystack, $needle) {
        return $needle !== '' && strpos($haystack, $needle) !== false;
    }
}

if (!function_exists('str_starts_with')) {
    function str_starts_with($haystack, $needle) {
        return $needle !== '' && strncmp($haystack, $needle, strlen($needle)) === 0;
    }
}

if (!function_exists('str_ends_with')) {
    function str_ends_with($haystack, $needle) {
        return $needle !== '' && substr($haystack, -strlen($needle)) === $needle;
    }
}

/**
 * PHP 8.1 compatible array_is_list function
 */
if (!function_exists('array_is_list')) {
    function array_is_list($array) {
        if (!is_array($array)) {
            return false;
        }
        
        if (empty($array)) {
            return true;
        }
        
        $keys = array_keys($array);
        return $keys === range(0, count($array) - 1);
    }
}

/**
 * PHP 8.1 compatible performance optimizations
 */
function firma_rehberi_php81_optimizations() {
    // Enable OPcache if available
    if (function_exists('opcache_get_status') && opcache_get_status()) {
        // OPcache is enabled, good for performance
    }
    
    // Set memory limit for better performance
    if (function_exists('ini_set')) {
        $current_limit = ini_get('memory_limit');
        $limit_bytes = wp_convert_hr_to_bytes($current_limit);
        
        // Increase memory limit if it's less than 256M
        if ($limit_bytes < wp_convert_hr_to_bytes('256M')) {
            ini_set('memory_limit', '256M');
        }
    }
}
add_action('init', 'firma_rehberi_php81_optimizations');

/**
 * PHP 8.1 compatible error handling
 */
function firma_rehberi_php81_error_handler($errno, $errstr, $errfile, $errline) {
    // Only handle errors in our theme files
    if (strpos($errfile, 'firma-rehberi-pro') === false) {
        return false;
    }
    
    // Log error for debugging
    error_log("Firma Rehberi Pro Error: $errstr in $errfile on line $errline");
    
    // Don't display errors in production
    if (defined('WP_DEBUG') && WP_DEBUG) {
        return false; // Let WordPress handle it
    }
    
    return true; // Suppress error
}

// Set custom error handler for theme files
set_error_handler('firma_rehberi_php81_error_handler', E_WARNING | E_NOTICE);

/**
 * PHP 8.1 compatible type checking
 */
function firma_rehberi_safe_get_post_meta($post_id, $meta_key, $single = true) {
    $value = get_post_meta($post_id, $meta_key, $single);
    
    // Ensure we return the expected type
    if ($single && is_array($value)) {
        return $value[0] ?? '';
    }
    
    return $value;
}

/**
 * PHP 8.1 compatible array operations
 */
function firma_rehberi_array_merge_recursive_distinct($array1, $array2) {
    $merged = $array1;
    
    foreach ($array2 as $key => $value) {
        if (is_array($value) && isset($merged[$key]) && is_array($merged[$key])) {
            $merged[$key] = firma_rehberi_array_merge_recursive_distinct($merged[$key], $value);
        } else {
            $merged[$key] = $value;
        }
    }
    
    return $merged;
}

/**
 * PHP 8.1 compatible JSON operations
 */
function firma_rehberi_safe_json_decode($json, $assoc = true, $depth = 512, $options = 0) {
    $decoded = json_decode($json, $assoc, $depth, $options);
    
    if (json_last_error() !== JSON_ERROR_NONE) {
        error_log('JSON Decode Error: ' . json_last_error_msg());
        return $assoc ? array() : null;
    }
    
    return $decoded;
}

/**
 * PHP 8.1 compatible string operations
 */
function firma_rehberi_safe_substr($string, $start, $length = null) {
    if (!is_string($string)) {
        return '';
    }
    
    if ($length === null) {
        return substr($string, $start);
    }
    
    return substr($string, $start, $length);
}

/**
 * PHP 8.1 compatible file operations
 */
function firma_rehberi_safe_file_get_contents($filename) {
    if (!file_exists($filename) || !is_readable($filename)) {
        return false;
    }
    
    $content = file_get_contents($filename);
    
    if ($content === false) {
        error_log("Failed to read file: $filename");
        return false;
    }
    
    return $content;
}

/**
 * PHP 8.1 compatible performance monitoring
 */
function firma_rehberi_php81_performance_monitor() {
    if (!defined('WP_DEBUG') || !WP_DEBUG) {
        return;
    }
    
    $memory_usage = memory_get_usage(true);
    $peak_memory = memory_get_peak_usage(true);
    $execution_time = microtime(true) - $_SERVER['REQUEST_TIME_FLOAT'];
    
    error_log(sprintf(
        'Firma Rehberi Pro Performance - Memory: %s, Peak: %s, Time: %.4fs',
        size_format($memory_usage),
        size_format($peak_memory),
        $execution_time
    ));
}
add_action('shutdown', 'firma_rehberi_php81_performance_monitor');
