<?php
/**
 * SEO Functions
 *
 * @package FirmaRehberiPro
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Generate SEO meta title
 */
function firma_rehberi_get_meta_title() {
    $title = '';
    
    if (is_front_page()) {
        $title = get_bloginfo('name');
        $description = get_bloginfo('description');
        if ($description) {
            $title .= ' - ' . $description;
        }
    } elseif (is_singular('company')) {
        $company_name = get_the_title();
        $city = '';
        $category = '';
        
        $cities = get_the_terms(get_the_ID(), 'company_city');
        if ($cities && !is_wp_error($cities)) {
            $city = $cities[0]->name;
        }
        
        $categories = get_the_terms(get_the_ID(), 'company_category');
        if ($categories && !is_wp_error($categories)) {
            $category = $categories[0]->name;
        }
        
        $title = $company_name;
        if ($category) {
            $title .= ' - ' . $category;
        }
        if ($city) {
            $title .= ' - ' . $city;
        }
        $title .= ' | ' . get_bloginfo('name');
    } elseif (is_post_type_archive('company')) {
        $title = __('Firmalar', 'firma-rehberi-pro') . ' | ' . get_bloginfo('name');
    } elseif (is_tax('company_category')) {
        $term = get_queried_object();
        $title = $term->name . ' ' . __('Firmaları', 'firma-rehberi-pro') . ' | ' . get_bloginfo('name');
    } elseif (is_tax('company_city')) {
        $term = get_queried_object();
        $title = $term->name . ' ' . __('Firmaları', 'firma-rehberi-pro') . ' | ' . get_bloginfo('name');
    } elseif (is_singular()) {
        $title = get_the_title() . ' | ' . get_bloginfo('name');
    } elseif (is_category()) {
        $title = single_cat_title('', false) . ' | ' . get_bloginfo('name');
    } elseif (is_tag()) {
        $title = single_tag_title('', false) . ' | ' . get_bloginfo('name');
    } elseif (is_search()) {
        $title = sprintf(__('"%s" için arama sonuçları', 'firma-rehberi-pro'), get_search_query()) . ' | ' . get_bloginfo('name');
    } elseif (is_404()) {
        $title = __('Sayfa Bulunamadı', 'firma-rehberi-pro') . ' | ' . get_bloginfo('name');
    } else {
        $title = get_bloginfo('name');
    }
    
    return apply_filters('firma_rehberi_meta_title', $title);
}

/**
 * Generate SEO meta description
 */
function firma_rehberi_get_meta_description() {
    $description = '';
    
    if (is_front_page()) {
        $description = get_bloginfo('description');
        if (!$description) {
            $description = __('Türkiye\'nin en kapsamlı firma rehberi. Binlerce işletmeyi keşfedin, detaylı bilgileri inceleyin ve en uygun hizmeti bulun.', 'firma-rehberi-pro');
        }
    } elseif (is_singular('company')) {
        $description = get_the_excerpt();
        if (!$description) {
            $description = get_the_title();
            $city = '';
            $category = '';
            
            $cities = get_the_terms(get_the_ID(), 'company_city');
            if ($cities && !is_wp_error($cities)) {
                $city = $cities[0]->name;
            }
            
            $categories = get_the_terms(get_the_ID(), 'company_category');
            if ($categories && !is_wp_error($categories)) {
                $category = $categories[0]->name;
            }
            
            if ($category && $city) {
                $description .= ' - ' . $category . ' hizmeti veren ' . $city . ' merkezli firma.';
            } elseif ($category) {
                $description .= ' - ' . $category . ' hizmeti veren firma.';
            } elseif ($city) {
                $description .= ' - ' . $city . ' merkezli firma.';
            }
        }
    } elseif (is_post_type_archive('company')) {
        $description = __('Türkiye\'nin en kapsamlı firma rehberi. Binlerce işletmeyi kategori ve şehir bazında filtreleyerek keşfedin.', 'firma-rehberi-pro');
    } elseif (is_tax('company_category')) {
        $term = get_queried_object();
        $description = sprintf(__('%s kategorisindeki firmaları keşfedin. Detaylı bilgileri inceleyin ve en uygun hizmeti bulun.', 'firma-rehberi-pro'), $term->name);
    } elseif (is_tax('company_city')) {
        $term = get_queried_object();
        $description = sprintf(__('%s şehrindeki firmaları keşfedin. Detaylı bilgileri inceleyin ve en uygun hizmeti bulun.', 'firma-rehberi-pro'), $term->name);
    } elseif (is_singular()) {
        $description = get_the_excerpt();
        if (!$description) {
            $description = get_the_title();
        }
    } elseif (is_category()) {
        $description = single_cat_title('', false) . ' kategorisindeki yazılar.';
    } elseif (is_tag()) {
        $description = single_tag_title('', false) . ' etiketli yazılar.';
    } elseif (is_search()) {
        $description = sprintf(__('"%s" için arama sonuçları. Aradığınızı bulamadınız mı? Farklı anahtar kelimeler deneyin.', 'firma-rehberi-pro'), get_search_query());
    } else {
        $description = get_bloginfo('description');
    }
    
    // Limit description length
    $description = wp_trim_words($description, 25, '...');
    
    return apply_filters('firma_rehberi_meta_description', $description);
}

/**
 * Add custom meta tags to head
 */
function firma_rehberi_add_meta_tags() {
    $title = firma_rehberi_get_meta_title();
    $description = firma_rehberi_get_meta_description();
    
    echo '<title>' . esc_html($title) . '</title>' . "\n";
    echo '<meta name="description" content="' . esc_attr($description) . '">' . "\n";
    
    // Add keywords
    if (is_singular('company')) {
        $keywords = array();
        
        $categories = get_the_terms(get_the_ID(), 'company_category');
        if ($categories && !is_wp_error($categories)) {
            foreach ($categories as $category) {
                $keywords[] = $category->name;
            }
        }
        
        $cities = get_the_terms(get_the_ID(), 'company_city');
        if ($cities && !is_wp_error($cities)) {
            foreach ($cities as $city) {
                $keywords[] = $city->name;
            }
        }
        
        $keywords[] = get_the_title();
        $keywords[] = 'firma';
        $keywords[] = 'işletme';
        
        if (!empty($keywords)) {
            echo '<meta name="keywords" content="' . esc_attr(implode(', ', $keywords)) . '">' . "\n";
        }
    }
    
    // Add author
    if (is_singular()) {
        echo '<meta name="author" content="' . esc_attr(get_bloginfo('name')) . '">' . "\n";
    }
    
    // Add last modified
    if (is_singular()) {
        $modified = get_the_modified_time('c');
        echo '<meta name="last-modified" content="' . esc_attr($modified) . '">' . "\n";
    }
}
add_action('wp_head', 'firma_rehberi_add_meta_tags', 1);

/**
 * Generate sitemap.xml
 */
function firma_rehberi_generate_sitemap() {
    if (isset($_GET['sitemap']) && $_GET['sitemap'] === 'xml') {
        header('Content-Type: application/xml; charset=utf-8');
        
        echo '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        echo '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";
        
        // Homepage
        echo '<url>' . "\n";
        echo '<loc>' . esc_url(home_url('/')) . '</loc>' . "\n";
        echo '<lastmod>' . date('c') . '</lastmod>' . "\n";
        echo '<changefreq>daily</changefreq>' . "\n";
        echo '<priority>1.0</priority>' . "\n";
        echo '</url>' . "\n";
        
        // Company archive
        echo '<url>' . "\n";
        echo '<loc>' . esc_url(get_post_type_archive_link('company')) . '</loc>' . "\n";
        echo '<lastmod>' . date('c') . '</lastmod>' . "\n";
        echo '<changefreq>daily</changefreq>' . "\n";
        echo '<priority>0.9</priority>' . "\n";
        echo '</url>' . "\n";
        
        // Individual companies
        $companies = get_posts(array(
            'post_type' => 'company',
            'posts_per_page' => -1,
            'post_status' => 'publish',
        ));
        
        foreach ($companies as $company) {
            echo '<url>' . "\n";
            echo '<loc>' . esc_url(get_permalink($company->ID)) . '</loc>' . "\n";
            echo '<lastmod>' . get_the_modified_time('c', $company->ID) . '</lastmod>' . "\n";
            echo '<changefreq>weekly</changefreq>' . "\n";
            echo '<priority>0.8</priority>' . "\n";
            echo '</url>' . "\n";
        }
        
        // Categories
        $categories = get_terms(array(
            'taxonomy' => 'company_category',
            'hide_empty' => true,
        ));
        
        foreach ($categories as $category) {
            echo '<url>' . "\n";
            echo '<loc>' . esc_url(get_term_link($category)) . '</loc>' . "\n";
            echo '<lastmod>' . date('c') . '</lastmod>' . "\n";
            echo '<changefreq>weekly</changefreq>' . "\n";
            echo '<priority>0.7</priority>' . "\n";
            echo '</url>' . "\n";
        }
        
        // Cities
        $cities = get_terms(array(
            'taxonomy' => 'company_city',
            'hide_empty' => true,
        ));
        
        foreach ($cities as $city) {
            echo '<url>' . "\n";
            echo '<loc>' . esc_url(get_term_link($city)) . '</loc>' . "\n";
            echo '<lastmod>' . date('c') . '</lastmod>' . "\n";
            echo '<changefreq>weekly</changefreq>' . "\n";
            echo '<priority>0.7</priority>' . "\n";
            echo '</url>' . "\n";
        }
        
        // Blog posts
        $posts = get_posts(array(
            'post_type' => 'post',
            'posts_per_page' => -1,
            'post_status' => 'publish',
        ));
        
        foreach ($posts as $post) {
            echo '<url>' . "\n";
            echo '<loc>' . esc_url(get_permalink($post->ID)) . '</loc>' . "\n";
            echo '<lastmod>' . get_the_modified_time('c', $post->ID) . '</lastmod>' . "\n";
            echo '<changefreq>monthly</changefreq>' . "\n";
            echo '<priority>0.6</priority>' . "\n";
            echo '</url>' . "\n";
        }
        
        echo '</urlset>';
        exit;
    }
}
add_action('init', 'firma_rehberi_generate_sitemap');

/**
 * Generate robots.txt
 */
function firma_rehberi_generate_robots() {
    if (strpos($_SERVER['REQUEST_URI'], '/robots.txt') !== false) {
        header('Content-Type: text/plain; charset=utf-8');
        
        echo "User-agent: *\n";
        echo "Allow: /\n";
        echo "Disallow: /wp-admin/\n";
        echo "Disallow: /wp-includes/\n";
        echo "Disallow: /wp-content/plugins/\n";
        echo "Disallow: /wp-content/themes/\n";
        echo "Disallow: /wp-content/cache/\n";
        echo "Disallow: /wp-content/uploads/\n";
        echo "Disallow: /wp-json/\n";
        echo "Disallow: /xmlrpc.php\n";
        echo "Disallow: /readme.html\n";
        echo "Disallow: /license.txt\n";
        echo "\n";
        echo "Sitemap: " . home_url('/?sitemap=xml') . "\n";
        
        exit;
    }
}
add_action('init', 'firma_rehberi_generate_robots');

/**
 * Add breadcrumb navigation
 */
function firma_rehberi_add_breadcrumbs() {
    if (!is_front_page()) {
        get_template_part('template-parts/breadcrumb');
    }
}
add_action('firma_rehberi_before_content', 'firma_rehberi_add_breadcrumbs');

/**
 * Optimize images for SEO
 */
function firma_rehberi_optimize_images($content) {
    if (is_singular('company')) {
        // Add alt text to images if missing
        $content = preg_replace_callback(
            '/<img([^>]*?)(?:\s+alt\s*=\s*["\']([^"\']*)["\'])?([^>]*?)>/i',
            function($matches) {
                $full_match = $matches[0];
                $before_alt = $matches[1];
                $alt_text = $matches[2];
                $after_alt = $matches[3];
                
                if (empty($alt_text)) {
                    $alt_text = get_the_title() . ' - ' . get_bloginfo('name');
                }
                
                return '<img' . $before_alt . ' alt="' . esc_attr($alt_text) . '"' . $after_alt . '>';
            },
            $content
        );
    }
    
    return $content;
}
add_filter('the_content', 'firma_rehberi_optimize_images');

/**
 * Add structured data for breadcrumbs
 */
function firma_rehberi_breadcrumb_schema() {
    if (is_front_page()) {
        return;
    }
    
    $breadcrumbs = array();
    $breadcrumbs[] = array(
        'name' => __('Ana Sayfa', 'firma-rehberi-pro'),
        'url'  => home_url('/'),
    );
    
    if (is_singular('company')) {
        $breadcrumbs[] = array(
            'name' => __('Firmalar', 'firma-rehberi-pro'),
            'url'  => get_post_type_archive_link('company'),
        );
        
        $categories = get_the_terms(get_the_ID(), 'company_category');
        if ($categories && !is_wp_error($categories)) {
            $breadcrumbs[] = array(
                'name' => $categories[0]->name,
                'url'  => get_term_link($categories[0]),
            );
        }
        
        $breadcrumbs[] = array(
            'name' => get_the_title(),
            'url'  => get_permalink(),
        );
    } elseif (is_post_type_archive('company')) {
        $breadcrumbs[] = array(
            'name' => __('Firmalar', 'firma-rehberi-pro'),
            'url'  => get_post_type_archive_link('company'),
        );
    } elseif (is_tax('company_category')) {
        $breadcrumbs[] = array(
            'name' => __('Firmalar', 'firma-rehberi-pro'),
            'url'  => get_post_type_archive_link('company'),
        );
        
        $term = get_queried_object();
        $breadcrumbs[] = array(
            'name' => $term->name,
            'url'  => get_term_link($term),
        );
    } elseif (is_tax('company_city')) {
        $breadcrumbs[] = array(
            'name' => __('Firmalar', 'firma-rehberi-pro'),
            'url'  => get_post_type_archive_link('company'),
        );
        
        $term = get_queried_object();
        $breadcrumbs[] = array(
            'name' => $term->name,
            'url'  => get_term_link($term),
        );
    }
    
    $schema = array(
        '@context' => 'https://schema.org',
        '@type' => 'BreadcrumbList',
        'itemListElement' => array()
    );
    
    foreach ($breadcrumbs as $index => $breadcrumb) {
        $schema['itemListElement'][] = array(
            '@type' => 'ListItem',
            'position' => $index + 1,
            'name' => $breadcrumb['name'],
            'item' => $breadcrumb['url']
        );
    }
    
    echo '<script type="application/ld+json">' . wp_json_encode($schema, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE) . '</script>' . "\n";
}
add_action('wp_head', 'firma_rehberi_breadcrumb_schema');
