'use client'

import { useState } from 'react'
import Link from 'next/link'
import { MapPin, Search, ArrowRight, Star, Users } from 'lucide-react'

export default function CitySearch() {
  const [searchQuery, setSearchQuery] = useState('')
  const [selectedRegion, setSelectedRegion] = useState('')

  const regions = [
    { name: 'Marmara', cities: ['İstanbul', 'Bursa', 'Balıkesir', 'Çanakkale', 'Edirne', 'Kırklareli', 'Kocaeli', 'Sakarya', 'Tekirdağ', 'Yalova'] },
    { name: 'Ege', cities: ['İzmir', 'Aydın', 'Denizli', 'Manisa', 'Muğla', 'Uşak'] },
    { name: 'Akdeniz', cities: ['Antalya', 'Adana', 'Mersin', 'Hatay', 'Kahramanmaraş', 'Osmaniye'] },
    { name: 'İç Anadolu', cities: ['Ankara', 'Konya', 'Kayseri', 'Sivas', 'Eskişehir', 'Aksaray', 'Çankırı', 'Çorum', 'Karaman', 'Kırıkkale', 'Kırşehir', 'Nevşehir', 'Niğde', 'Yozgat'] },
    { name: 'Karadeniz', cities: ['Samsun', 'Trabzon', 'Ordu', 'Giresun', 'Rize', 'Artvin', 'Gümüşhane', 'Bayburt', 'Kastamonu', 'Sinop', 'Amasya', 'Tokat', 'Çorum', 'Zonguldak', 'Bartın', 'Karabük', 'Düzce'] },
    { name: 'Doğu Anadolu', cities: ['Erzurum', 'Erzincan', 'Malatya', 'Elazığ', 'Tunceli', 'Bingöl', 'Muş', 'Bitlis', 'Van', 'Hakkari', 'Ağrı', 'Kars', 'Iğdır', 'Ardahan'] },
    { name: 'Güneydoğu Anadolu', cities: ['Gaziantep', 'Şanlıurfa', 'Diyarbakır', 'Mardin', 'Batman', 'Siirt', 'Şırnak', 'Adıyaman', 'Kilis'] }
  ]

  const popularCities = [
    { name: 'İstanbul', courseCount: 156, image: 'https://images.unsplash.com/photo-1527838832700-5059252407fa?w=600&h=400&fit=crop', region: 'Marmara', color: 'from-blue-600 to-blue-800' },
    { name: 'Ankara', courseCount: 89, image: 'https://images.unsplash.com/photo-1583417319070-4a69db38a482?w=600&h=400&fit=crop', region: 'İç Anadolu', color: 'from-gray-600 to-gray-800' },
    { name: 'İzmir', courseCount: 67, image: 'https://images.unsplash.com/photo-1541432901042-2d8bd64b4a9b?w=600&h=400&fit=crop', region: 'Ege', color: 'from-cyan-600 to-cyan-800' },
    { name: 'Bursa', courseCount: 45, image: 'https://images.unsplash.com/photo-1605649487212-47bdab064df7?w=600&h=400&fit=crop', region: 'Marmara', color: 'from-green-600 to-green-800' },
    { name: 'Antalya', courseCount: 38, image: 'https://images.unsplash.com/photo-1566073771259-6a8506099945?w=600&h=400&fit=crop', region: 'Akdeniz', color: 'from-orange-600 to-orange-800' },
    { name: 'Adana', courseCount: 32, image: 'https://images.unsplash.com/photo-1519817914152-22d216bb9170?w=600&h=400&fit=crop', region: 'Akdeniz', color: 'from-red-600 to-red-800' },
    { name: 'Konya', courseCount: 28, image: 'https://images.unsplash.com/photo-1609137144813-7d9921338f24?w=600&h=400&fit=crop', region: 'İç Anadolu', color: 'from-indigo-600 to-indigo-800' },
    { name: 'Gaziantep', courseCount: 25, image: 'https://images.unsplash.com/photo-1541480601022-2308c0f02487?w=600&h=400&fit=crop', region: 'Güneydoğu Anadolu', color: 'from-amber-600 to-amber-800' }
  ]

  const filteredCities = popularCities.filter(city => {
    const matchesSearch = city.name.toLowerCase().includes(searchQuery.toLowerCase())
    const matchesRegion = !selectedRegion || city.region === selectedRegion
    return matchesSearch && matchesRegion
  })

  return (
    <section className="section-padding bg-gray-50">
      <div className="container-custom">
        <div className="text-center mb-16">
          <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-6">
            Şehir Bazında Dans Kursları
          </h2>
          <p className="text-xl text-gray-600 max-w-3xl mx-auto">
            Yaşadığınız şehirdeki en iyi dans kurslarını keşfedin. 
            İstanbul, Ankara, İzmir ve daha fazla şehirde profesyonel eğitim.
          </p>
        </div>

        {/* Search and Filter */}
        <div className="max-w-4xl mx-auto mb-12">
          <div className="bg-white rounded-2xl p-6 shadow-lg">
            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
              {/* Search Input */}
              <div className="relative">
                <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
                <input
                  type="text"
                  placeholder="Şehir ara..."
                  value={searchQuery}
                  onChange={(e) => setSearchQuery(e.target.value)}
                  className="w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent"
                />
              </div>

              {/* Region Filter */}
              <div className="relative">
                <MapPin className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
                <select
                  value={selectedRegion}
                  onChange={(e) => setSelectedRegion(e.target.value)}
                  className="w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent appearance-none bg-white"
                >
                  <option value="">Tüm Bölgeler</option>
                  {regions.map((region) => (
                    <option key={region.name} value={region.name}>
                      {region.name} Bölgesi
                    </option>
                  ))}
                </select>
              </div>
            </div>
          </div>
        </div>

        {/* Popular Cities Grid */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-12">
          {filteredCities.map((city, index) => (
            <Link
              key={city.name}
              href={`/sehirler/${city.name.toLowerCase()}`}
              className="group card-hover"
            >
              <div className="relative overflow-hidden rounded-t-xl">
                <img
                  src={city.image}
                  alt={`${city.name} dans kursları`}
                  loading="lazy"
                  className="w-full h-40 object-cover group-hover:scale-110 transition-transform duration-500"
                />
                <div className={`absolute inset-0 bg-gradient-to-br ${city.color} opacity-40 group-hover:opacity-30 transition-opacity duration-300`}></div>
                <div className="absolute inset-0 bg-gradient-to-t from-black/80 via-black/20 to-transparent"></div>
                
                {/* City Name */}
                <div className="absolute bottom-4 left-4 right-4">
                  <h3 className="text-white text-xl font-bold mb-1">{city.name}</h3>
                  <p className="text-gray-200 text-sm">{city.region} Bölgesi</p>
                </div>

                {/* Course Count Badge */}
                <div className="absolute top-4 right-4">
                  <div className="bg-white/90 text-gray-800 px-3 py-1 rounded-full text-sm font-medium">
                    {city.courseCount} kurs
                  </div>
                </div>
              </div>

              <div className="p-4">
                <div className="flex items-center justify-between text-sm text-gray-500">
                  <div className="flex items-center space-x-1">
                    <Users className="w-4 h-4" />
                    <span>{city.courseCount} kurs mevcut</span>
                  </div>
                  <div className="flex items-center space-x-1 text-primary-600 group-hover:text-primary-700">
                    <span>Detayları Gör</span>
                    <ArrowRight className="w-4 h-4 group-hover:translate-x-1 transition-transform" />
                  </div>
                </div>
              </div>
            </Link>
          ))}
        </div>

        {/* All Regions */}
        <div className="bg-white rounded-2xl p-8 shadow-lg">
          <h3 className="text-2xl font-bold text-gray-900 mb-6 text-center">
            Tüm Bölgeler
          </h3>
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-6">
            {regions.map((region, index) => (
              <div key={region.name} className="border border-gray-200 rounded-lg p-4 hover:shadow-md transition-shadow">
                <h4 className="font-semibold text-gray-900 mb-3">{region.name} Bölgesi</h4>
                <div className="space-y-1">
                  {region.cities.slice(0, 6).map((city) => (
                    <Link
                      key={city}
                      href={`/sehirler/${city.toLowerCase()}`}
                      className="block text-sm text-gray-600 hover:text-primary-600 transition-colors"
                    >
                      {city}
                    </Link>
                  ))}
                  {region.cities.length > 6 && (
                    <span className="text-xs text-gray-400">
                      +{region.cities.length - 6} şehir daha
                    </span>
                  )}
                </div>
              </div>
            ))}
          </div>
        </div>

        {/* CTA */}
        <div className="text-center mt-12">
          <Link
            href="/sehirler"
            className="btn-primary text-lg px-8 py-3 inline-flex items-center space-x-2"
          >
            <span>Tüm Şehirleri Gör</span>
            <ArrowRight className="w-5 h-5" />
          </Link>
        </div>
      </div>
    </section>
  )
}
