'use client'

import { useState } from 'react'
import Link from 'next/link'
import { Star, MapPin, Clock, Users, Phone, ArrowRight, Heart } from 'lucide-react'

export default function FeaturedCourses() {
  const [favorites, setFavorites] = useState<number[]>([])

  const toggleFavorite = (courseId: number) => {
    setFavorites(prev => 
      prev.includes(courseId) 
        ? prev.filter(id => id !== courseId)
        : [...prev, courseId]
    )
  }

  const featuredCourses = [
    {
      id: 1,
      name: 'Salsa & Bachata Akademi',
      location: 'Kadıköy, İstanbul',
      danceTypes: ['Salsa', 'Bachata', 'Kizomba'],
      rating: 4.9,
      reviewCount: 127,
      price: '₺150',
      duration: '8 hafta',
      level: 'Başlangıç',
      schedule: 'Pazartesi, Çarşamba 19:00-21:00',
      image: 'https://images.unsplash.com/photo-1547036967-23d11aacaee0?w=400&h=300&fit=crop',
      description: 'Profesyonel Latin dans eğitimi. Her seviyeye uygun grup dersleri.',
      instructor: 'Maria Rodriguez',
      phone: '+90 532 123 4567',
      features: ['Profesyonel Öğretmen', 'Küçük Gruplar', 'Sertifika', 'Sosyal Etkinlikler']
    },
    {
      id: 2,
      name: 'Modern Dans Stüdyosu',
      location: 'Çankaya, Ankara',
      danceTypes: ['Modern Dans', 'Contemporary', 'Jazz'],
      rating: 4.8,
      reviewCount: 89,
      price: '₺120',
      duration: '10 hafta',
      level: 'Orta',
      schedule: 'Salı, Perşembe 18:30-20:30',
      image: 'https://images.unsplash.com/photo-1518611012118-696072aa579a?w=400&h=300&fit=crop',
      description: 'Yaratıcılığınızı keşfedin. Modern dans teknikleri ve koreografi.',
      instructor: 'Ahmet Yılmaz',
      phone: '+90 312 987 6543',
      features: ['Yaratıcı Koreografi', 'Sahne Deneyimi', 'Grup Performansları', 'Kişisel Gelişim']
    },
    {
      id: 3,
      name: 'Tango Passion',
      location: 'Konak, İzmir',
      danceTypes: ['Tango', 'Milonga', 'Vals'],
      rating: 4.9,
      reviewCount: 156,
      price: '₺180',
      duration: '12 hafta',
      level: 'Tüm Seviyeler',
      schedule: 'Cumartesi, Pazar 15:00-17:00',
      image: 'https://images.unsplash.com/photo-1508700929628-666bc8bd84ea?w=400&h=300&fit=crop',
      description: 'Tutkulu Arjantin tangosu. Romantik ve zarif dans deneyimi.',
      instructor: 'Carlos & Elena',
      phone: '+90 232 555 7890',
      features: ['Çift Eğitimi', 'Müzik Eğitimi', 'Sosyal Danslar', 'Yarışma Hazırlığı']
    },
    {
      id: 4,
      name: 'Hip-Hop Academy',
      location: 'Beşiktaş, İstanbul',
      danceTypes: ['Hip-Hop', 'Breakdance', 'Popping'],
      rating: 4.7,
      reviewCount: 203,
      price: '₺100',
      duration: '6 hafta',
      level: 'Başlangıç',
      schedule: 'Pazartesi, Çarşamba, Cuma 20:00-22:00',
      image: 'https://images.unsplash.com/photo-1518611012118-696072aa579a?w=400&h=300&fit=crop',
      description: 'Şehir kültürünün en enerjik dansı. Genç ve dinamik eğitim.',
      instructor: 'DJ Kaan',
      phone: '+90 532 456 7890',
      features: ['Freestyle', 'Koreografi', 'Müzik Prodüksiyon', 'Sosyal Medya']
    },
    {
      id: 5,
      name: 'Bale & Dans Okulu',
      location: 'Nilüfer, Bursa',
      danceTypes: ['Klasik Bale', 'Modern Bale', 'Karakter'],
      rating: 4.9,
      reviewCount: 78,
      price: '₺200',
      duration: '16 hafta',
      level: 'Tüm Seviyeler',
      schedule: 'Salı, Perşembe, Cumartesi 16:00-18:00',
      image: 'https://images.unsplash.com/photo-1508700929628-666bc8bd84ea?w=400&h=300&fit=crop',
      description: 'Klasik bale eğitimi. Disiplinli ve zarif dans öğrenimi.',
      instructor: 'Natalia Petrov',
      phone: '+90 224 333 4567',
      features: ['Rus Metodu', 'Sahne Sanatları', 'Kostüm Tasarım', 'Yarışma Takımı']
    },
    {
      id: 6,
      name: 'Latin Fire Dance',
      location: 'Muratpaşa, Antalya',
      danceTypes: ['Salsa', 'Bachata', 'Merengue', 'Cha Cha'],
      rating: 4.8,
      reviewCount: 145,
      price: '₺130',
      duration: '8 hafta',
      level: 'Başlangıç',
      schedule: 'Pazartesi, Çarşamba, Cuma 19:30-21:30',
      image: 'https://images.unsplash.com/photo-1571019613454-1cb2f99b2d8b?w=400&h=300&fit=crop',
      description: 'Sıcak Latin ritimleri. Eğlenceli ve sosyal dans deneyimi.',
      instructor: 'Isabella Cruz',
      phone: '+90 242 777 8888',
      features: ['Sosyal Danslar', 'Sahne Performansı', 'Müzik Eğitimi', 'Tatil Paketleri']
    }
  ]

  return (
    <section className="section-padding bg-white">
      <div className="container-custom">
        <div className="text-center mb-16">
          <h2 className="text-4xl md:text-5xl font-bold text-gray-900 mb-6">
            Öne Çıkan Dans Kursları
          </h2>
          <p className="text-xl text-gray-600 max-w-3xl mx-auto">
            En popüler ve en çok tercih edilen dans kurslarını keşfedin. 
            Profesyonel öğretmenler, kaliteli eğitim ve uygun fiyatlar.
          </p>
        </div>

        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-8">
          {featuredCourses.map((course, index) => (
            <div key={course.id} className="card-hover group">
              {/* Image */}
              <div className="relative overflow-hidden rounded-t-xl">
                <img
                  src={course.image}
                  alt={course.name}
                  className="w-full h-48 object-cover group-hover:scale-105 transition-transform duration-300"
                />
                <div className="absolute inset-0 bg-gradient-to-t from-black/60 to-transparent"></div>
                
                {/* Favorite Button */}
                <button
                  onClick={() => toggleFavorite(course.id)}
                  className="absolute top-4 right-4 p-2 bg-white/90 rounded-full hover:bg-white transition-colors"
                >
                  <Heart 
                    className={`w-5 h-5 ${
                      favorites.includes(course.id) 
                        ? 'text-red-500 fill-current' 
                        : 'text-gray-400 hover:text-red-500'
                    }`} 
                  />
                </button>

                {/* Price Badge */}
                <div className="absolute bottom-4 left-4">
                  <div className="bg-primary-600 text-white px-3 py-1 rounded-full text-sm font-bold">
                    {course.price}/ay
                  </div>
                </div>

                {/* Level Badge */}
                <div className="absolute top-4 left-4">
                  <div className="bg-white/90 text-gray-800 px-3 py-1 rounded-full text-sm font-medium">
                    {course.level}
                  </div>
                </div>
              </div>

              {/* Content */}
              <div className="p-6">
                {/* Header */}
                <div className="mb-4">
                  <h3 className="text-xl font-bold text-gray-900 mb-2 group-hover:text-primary-600 transition-colors">
                    {course.name}
                  </h3>
                  <div className="flex items-center text-gray-600 mb-2">
                    <MapPin className="w-4 h-4 mr-1" />
                    <span className="text-sm">{course.location}</span>
                  </div>
                  <div className="flex items-center space-x-1 mb-3">
                    <Star className="w-4 h-4 text-yellow-400 fill-current" />
                    <span className="text-sm font-medium">{course.rating}</span>
                    <span className="text-sm text-gray-500">({course.reviewCount} değerlendirme)</span>
                  </div>
                </div>

                {/* Dance Types */}
                <div className="mb-4">
                  <div className="flex flex-wrap gap-2">
                    {course.danceTypes.map((type) => (
                      <span
                        key={type}
                        className="bg-primary-100 text-primary-800 px-2 py-1 rounded-full text-xs font-medium"
                      >
                        {type}
                      </span>
                    ))}
                  </div>
                </div>

                {/* Description */}
                <p className="text-gray-600 text-sm mb-4 line-clamp-2">
                  {course.description}
                </p>

                {/* Features */}
                <div className="mb-4">
                  <div className="grid grid-cols-2 gap-2">
                    {course.features.slice(0, 2).map((feature) => (
                      <div key={feature} className="flex items-center text-xs text-gray-600">
                        <div className="w-1.5 h-1.5 bg-primary-500 rounded-full mr-2"></div>
                        {feature}
                      </div>
                    ))}
                  </div>
                </div>

                {/* Schedule & Duration */}
                <div className="flex items-center justify-between text-sm text-gray-500 mb-4">
                  <div className="flex items-center space-x-1">
                    <Clock className="w-4 h-4" />
                    <span>{course.duration}</span>
                  </div>
                  <div className="flex items-center space-x-1">
                    <Users className="w-4 h-4" />
                    <span>{course.schedule}</span>
                  </div>
                </div>

                {/* Instructor */}
                <div className="text-sm text-gray-600 mb-4">
                  <span className="font-medium">Eğitmen:</span> {course.instructor}
                </div>

                {/* Actions */}
                <div className="flex space-x-2">
                  <Link
                    href={`/kurslar/${course.id}`}
                    className="flex-1 btn-primary text-center text-sm py-2"
                  >
                    Detayları Gör
                  </Link>
                  <a
                    href={`tel:${course.phone}`}
                    className="flex items-center justify-center px-4 py-2 border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors"
                  >
                    <Phone className="w-4 h-4" />
                  </a>
                </div>
              </div>
            </div>
          ))}
        </div>

        {/* CTA */}
        <div className="text-center mt-12">
          <Link
            href="/kurslar"
            className="btn-primary text-lg px-8 py-3 inline-flex items-center space-x-2"
          >
            <span>Tüm Kursları Gör</span>
            <ArrowRight className="w-5 h-5" />
          </Link>
        </div>
      </div>
    </section>
  )
}

