import { Metadata } from 'next'

/**
 * SEOHead Component
 * This component helps generate comprehensive SEO metadata for pages
 * 
 * Usage:
 * export const metadata = generateSEOMetadata({
 *   title: 'Page Title',
 *   description: 'Page description',
 *   path: '/page-path',
 * })
 */

interface SEOMetadataProps {
  title: string
  description: string
  path: string
  keywords?: string[]
  image?: string
  type?: 'website' | 'article'
  noindex?: boolean
  author?: string
  publishedTime?: string
  modifiedTime?: string
}

export function generateSEOMetadata(props: SEOMetadataProps): Metadata {
  const {
    title,
    description,
    path,
    keywords = [],
    image = '/og-image.jpg',
    type = 'website',
    noindex = false,
    author,
    publishedTime,
    modifiedTime,
  } = props

  const baseUrl = process.env.NEXT_PUBLIC_BASE_URL || 'https://dansdersleri.com'
  const fullUrl = `${baseUrl}${path}`
  const fullImageUrl = image.startsWith('http') ? image : `${baseUrl}${image}`

  const metadata: Metadata = {
    title,
    description,
    keywords: keywords.join(', '),
    authors: author ? [{ name: author }] : [{ name: 'Dans Dersleri' }],
    creator: 'Dans Dersleri',
    publisher: 'Dans Dersleri',
    alternates: {
      canonical: fullUrl,
    },
    openGraph: {
      title,
      description,
      url: fullUrl,
      siteName: 'Dans Dersleri',
      locale: 'tr_TR',
      type,
      images: [
        {
          url: fullImageUrl,
          width: 1200,
          height: 630,
          alt: title,
        },
      ],
    },
    twitter: {
      card: 'summary_large_image',
      title,
      description,
      images: [fullImageUrl],
      creator: '@dansdersleri',
    },
    robots: noindex
      ? {
          index: false,
          follow: true,
        }
      : {
          index: true,
          follow: true,
          googleBot: {
            index: true,
            follow: true,
            'max-video-preview': -1,
            'max-image-preview': 'large',
            'max-snippet': -1,
          },
        },
  }

  // Add article-specific metadata
  if (type === 'article') {
    metadata.openGraph = {
      ...metadata.openGraph,
      type: 'article',
      publishedTime,
      modifiedTime,
      authors: author ? [author] : undefined,
    }
  }

  return metadata
}

/**
 * Quick metadata generators for common page types
 */

// Home page
export const homePageMetadata: Metadata = generateSEOMetadata({
  title: 'Dans Dersleri - Türkiye\'nin En Kapsamlı Dans Kursu Rehberi',
  description:
    'İstanbul, Ankara, İzmir ve tüm Türkiye\'de salsa, bachata, tango, hip-hop ve daha fazla dans türü için en iyi kursları keşfedin. Profesyonel öğretmenler, uygun fiyatlar.',
  path: '/',
  keywords: [
    'dans kursu',
    'dans dersleri',
    'salsa',
    'bachata',
    'tango',
    'hip-hop',
    'istanbul dans',
    'ankara dans',
    'izmir dans',
  ],
})

// Search page
export const searchPageMetadata: Metadata = generateSEOMetadata({
  title: 'Kurs Ara - Dans Dersleri',
  description: 'Türkiye\'nin en kapsamlı dans kursu rehberinde aradığınız kursu bulun.',
  path: '/arama',
  noindex: true,
})

// About page
export const aboutPageMetadata: Metadata = generateSEOMetadata({
  title: 'Hakkımızda - Dans Dersleri',
  description: 'Dans Dersleri hakkında bilgi edinin. Misyonumuz, vizyonumuz ve ekibimiz.',
  path: '/hakkimizda',
})

// Contact page
export const contactPageMetadata: Metadata = generateSEOMetadata({
  title: 'İletişim - Dans Dersleri',
  description: 'Dans Dersleri ile iletişime geçin. Sorularınız için bizimle iletişime geçebilirsiniz.',
  path: '/iletisim',
})

