'use client'

import { useState } from 'react'
import Link from 'next/link'
import Image from 'next/image'
import { Eye, Edit, Trash2, Star, MapPin, MoreVertical, CheckCircle, XCircle } from 'lucide-react'

export default function CourseTable() {
  const [selectedCourses, setSelectedCourses] = useState<number[]>([])

  const courses = [
    {
      id: 1,
      name: 'Salsa & Bachata Akademi',
      image: 'https://images.unsplash.com/photo-1547036967-23d11aacaee0?w=100&h=100&fit=crop',
      city: 'Kadıköy, İstanbul',
      instructor: 'Maria Rodriguez',
      danceTypes: ['Salsa', 'Bachata'],
      rating: 4.9,
      students: 127,
      price: 150,
      status: 'active',
      package: 'VIP',
      date: '2024-01-15'
    },
    {
      id: 2,
      name: 'Modern Dans Stüdyosu',
      image: 'https://images.unsplash.com/photo-1518611012118-696072aa579a?w=100&h=100&fit=crop',
      city: 'Çankaya, Ankara',
      instructor: 'Ahmet Yılmaz',
      danceTypes: ['Modern Dans', 'Contemporary'],
      rating: 4.8,
      students: 89,
      price: 120,
      status: 'active',
      package: 'Premium',
      date: '2024-01-14'
    },
    {
      id: 3,
      name: 'Tango Passion',
      image: 'https://images.unsplash.com/photo-1508700929628-666bc8bd84ea?w=100&h=100&fit=crop',
      city: 'Konak, İzmir',
      instructor: 'Carlos & Elena',
      danceTypes: ['Tango', 'Milonga'],
      rating: 4.9,
      students: 156,
      price: 180,
      status: 'pending',
      package: 'VIP',
      date: '2024-01-13'
    },
    {
      id: 4,
      name: 'Hip-Hop Academy',
      image: 'https://images.unsplash.com/photo-1518611012118-696072aa579a?w=100&h=100&fit=crop',
      city: 'Beşiktaş, İstanbul',
      instructor: 'DJ Kaan',
      danceTypes: ['Hip-Hop', 'Breakdance'],
      rating: 4.7,
      students: 203,
      price: 100,
      status: 'active',
      package: 'Premium',
      date: '2024-01-12'
    },
    {
      id: 5,
      name: 'Bale & Dans Okulu',
      image: 'https://images.unsplash.com/photo-1508700929628-666bc8bd84ea?w=100&h=100&fit=crop',
      city: 'Nilüfer, Bursa',
      instructor: 'Natalia Petrov',
      danceTypes: ['Klasik Bale', 'Modern Bale'],
      rating: 4.9,
      students: 78,
      price: 200,
      status: 'active',
      package: 'Basic',
      date: '2024-01-11'
    }
  ]

  const toggleSelectCourse = (id: number) => {
    setSelectedCourses(prev =>
      prev.includes(id) ? prev.filter(cid => cid !== id) : [...prev, id]
    )
  }

  const toggleSelectAll = () => {
    setSelectedCourses(prev =>
      prev.length === courses.length ? [] : courses.map(c => c.id)
    )
  }

  const getStatusBadge = (status: string) => {
    switch (status) {
      case 'active':
        return <span className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-green-100 text-green-800">
          <CheckCircle className="w-3 h-3 mr-1" />
          Aktif
        </span>
      case 'pending':
        return <span className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800">
          Onay Bekliyor
        </span>
      case 'inactive':
        return <span className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-red-100 text-red-800">
          <XCircle className="w-3 h-3 mr-1" />
          Pasif
        </span>
      default:
        return null
    }
  }

  const getPackageBadge = (pkg: string) => {
    switch (pkg) {
      case 'VIP':
        return <span className="px-2 py-0.5 rounded-full text-xs font-medium bg-purple-100 text-purple-800">VIP</span>
      case 'Premium':
        return <span className="px-2 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">Premium</span>
      case 'Basic':
        return <span className="px-2 py-0.5 rounded-full text-xs font-medium bg-gray-100 text-gray-800">Basic</span>
      default:
        return null
    }
  }

  return (
    <div className="bg-white rounded-xl shadow-lg overflow-hidden">
      {/* Bulk Actions */}
      {selectedCourses.length > 0 && (
        <div className="bg-primary-50 border-b border-primary-200 px-6 py-4">
          <div className="flex items-center justify-between">
            <span className="text-sm text-primary-900 font-medium">
              {selectedCourses.length} kurs seçildi
            </span>
            <div className="flex items-center space-x-2">
              <button className="px-4 py-2 text-sm bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors">
                Toplu Düzenle
              </button>
              <button className="px-4 py-2 text-sm bg-red-600 hover:bg-red-700 text-white rounded-lg transition-colors">
                Toplu Sil
              </button>
            </div>
          </div>
        </div>
      )}

      <div className="overflow-x-auto">
        <table className="w-full">
          <thead className="bg-gray-50 border-b border-gray-200">
            <tr>
              <th className="px-6 py-4 text-left">
                <input
                  type="checkbox"
                  checked={selectedCourses.length === courses.length}
                  onChange={toggleSelectAll}
                  className="w-4 h-4 text-primary-600 rounded focus:ring-primary-500"
                />
              </th>
              <th className="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase">Kurs</th>
              <th className="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase">Öğretmen</th>
              <th className="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase">Dans Türleri</th>
              <th className="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase">Puan</th>
              <th className="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase">Öğrenci</th>
              <th className="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase">Fiyat</th>
              <th className="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase">Paket</th>
              <th className="px-6 py-4 text-left text-xs font-semibold text-gray-600 uppercase">Durum</th>
              <th className="px-6 py-4 text-right text-xs font-semibold text-gray-600 uppercase">İşlemler</th>
            </tr>
          </thead>
          <tbody>
            {courses.map((course) => (
              <tr key={course.id} className="border-b border-gray-100 hover:bg-gray-50">
                <td className="px-6 py-4">
                  <input
                    type="checkbox"
                    checked={selectedCourses.includes(course.id)}
                    onChange={() => toggleSelectCourse(course.id)}
                    className="w-4 h-4 text-primary-600 rounded focus:ring-primary-500"
                  />
                </td>
                <td className="px-6 py-4">
                  <div className="flex items-center space-x-3">
                    <Image
                      src={course.image}
                      alt={course.name}
                      width={48}
                      height={48}
                      className="rounded-lg object-cover"
                    />
                    <div>
                      <p className="text-sm font-medium text-gray-900">{course.name}</p>
                      <p className="text-xs text-gray-500 flex items-center space-x-1">
                        <MapPin className="w-3 h-3" />
                        <span>{course.city}</span>
                      </p>
                    </div>
                  </div>
                </td>
                <td className="px-6 py-4">
                  <p className="text-sm text-gray-900">{course.instructor}</p>
                </td>
                <td className="px-6 py-4">
                  <div className="flex flex-wrap gap-1">
                    {course.danceTypes.map((type, index) => (
                      <span key={index} className="px-2 py-0.5 bg-primary-100 text-primary-800 text-xs rounded-full">
                        {type}
                      </span>
                    ))}
                  </div>
                </td>
                <td className="px-6 py-4">
                  <div className="flex items-center space-x-1">
                    <Star className="w-4 h-4 text-yellow-400 fill-current" />
                    <span className="text-sm font-medium text-gray-900">{course.rating}</span>
                  </div>
                </td>
                <td className="px-6 py-4">
                  <p className="text-sm text-gray-900">{course.students}</p>
                </td>
                <td className="px-6 py-4">
                  <p className="text-sm font-medium text-gray-900">₺{course.price}</p>
                </td>
                <td className="px-6 py-4">
                  {getPackageBadge(course.package)}
                </td>
                <td className="px-6 py-4">
                  {getStatusBadge(course.status)}
                </td>
                <td className="px-6 py-4">
                  <div className="flex items-center justify-end space-x-2">
                    <Link
                      href={`/kurslar/${course.id}`}
                      target="_blank"
                      className="p-2 text-gray-400 hover:text-blue-600 transition-colors"
                    >
                      <Eye className="w-4 h-4" />
                    </Link>
                    <Link
                      href={`/admin/kurslar/${course.id}/duzenle`}
                      className="p-2 text-gray-400 hover:text-green-600 transition-colors"
                    >
                      <Edit className="w-4 h-4" />
                    </Link>
                    <button className="p-2 text-gray-400 hover:text-red-600 transition-colors">
                      <Trash2 className="w-4 h-4" />
                    </button>
                    <button className="p-2 text-gray-400 hover:text-gray-600 transition-colors">
                      <MoreVertical className="w-4 h-4" />
                    </button>
                  </div>
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>

      {/* Pagination */}
      <div className="px-6 py-4 border-t border-gray-200">
        <div className="flex items-center justify-between">
          <p className="text-sm text-gray-600">
            Toplam <span className="font-medium">486</span> kurs - Sayfa <span className="font-medium">1</span> / <span className="font-medium">49</span>
          </p>
          <div className="flex items-center space-x-2">
            <button className="px-4 py-2 text-sm border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors disabled:opacity-50" disabled>
              Önceki
            </button>
            <button className="px-4 py-2 text-sm bg-primary-600 text-white rounded-lg">1</button>
            <button className="px-4 py-2 text-sm border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors">2</button>
            <button className="px-4 py-2 text-sm border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors">3</button>
            <button className="px-4 py-2 text-sm border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors">...</button>
            <button className="px-4 py-2 text-sm border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors">49</button>
            <button className="px-4 py-2 text-sm border border-gray-300 rounded-lg hover:bg-gray-50 transition-colors">
              Sonraki
            </button>
          </div>
        </div>
      </div>
    </div>
  )
}
