'use client'

import { useState } from 'react'
import { useRouter } from 'next/navigation'
import { Lock, Mail, Eye, EyeOff, AlertCircle, Shield } from 'lucide-react'

export default function LoginForm() {
  const router = useRouter()
  const [showPassword, setShowPassword] = useState(false)
  const [loading, setLoading] = useState(false)
  const [error, setError] = useState('')
  const [loginAttempts, setLoginAttempts] = useState(0)
  const [isLocked, setIsLocked] = useState(false)
  
  const [formData, setFormData] = useState({
    email: '',
    password: '',
    rememberMe: false
  })

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault()
    setError('')
    setLoading(true)

    // Rate limiting - 5 deneme sonrası kilitle
    if (loginAttempts >= 5) {
      setIsLocked(true)
      setError('Çok fazla başarısız giriş denemesi. Lütfen 15 dakika sonra tekrar deneyin.')
      setLoading(false)
      return
    }

    try {
      // Simüle edilmiş login - Gerçek uygulamada API çağrısı yapılacak
      await new Promise(resolve => setTimeout(resolve, 1000))

      // Demo hesaplar
      const validCredentials = [
        { email: 'admin@dansdersleri.com', password: 'DansDers2024!Admin', role: 'super_admin' },
        { email: 'editor@dansdersleri.com', password: 'DansDers2024!Edit', role: 'editor' },
        { email: 'moderator@dansdersleri.com', password: 'DansDers2024!Mod', role: 'moderator' }
      ]

      const user = validCredentials.find(
        cred => cred.email === formData.email && cred.password === formData.password
      )

      if (user) {
        // Session oluştur
        const sessionData = {
          user: {
            email: user.email,
            role: user.role,
            name: user.email.split('@')[0]
          },
          token: generateToken(),
          expiresAt: new Date(Date.now() + 24 * 60 * 60 * 1000).toISOString()
        }

        // LocalStorage'a kaydet
        localStorage.setItem('adminSession', JSON.stringify(sessionData))
        
        // Cookie'ye de kaydet (middleware için)
        document.cookie = `adminSession=${JSON.stringify(sessionData)}; path=/; max-age=${24 * 60 * 60}; SameSite=Strict`
        
        // Admin panele yönlendir
        setTimeout(() => {
          window.location.href = '/admin'
        }, 100)
      } else {
        setLoginAttempts(prev => prev + 1)
        setError('E-posta veya şifre hatalı. Lütfen tekrar deneyin.')
      }
    } catch (err) {
      setError('Bir hata oluştu. Lütfen tekrar deneyin.')
    } finally {
      setLoading(false)
    }
  }

  const generateToken = () => {
    return Math.random().toString(36).substring(2) + Date.now().toString(36)
  }

  return (
    <div className="bg-white rounded-2xl shadow-2xl p-8">
      {/* Error Message */}
      {error && (
        <div className="mb-6 p-4 bg-red-50 border border-red-200 rounded-lg flex items-start space-x-2">
          <AlertCircle className="w-5 h-5 text-red-600 mt-0.5" />
          <div className="flex-1">
            <p className="text-sm text-red-800">{error}</p>
            {loginAttempts > 0 && !isLocked && (
              <p className="text-xs text-red-600 mt-1">
                Kalan deneme hakkı: {5 - loginAttempts}
              </p>
            )}
          </div>
        </div>
      )}

      {/* Login Form */}
      <form onSubmit={handleSubmit} className="space-y-6">
        {/* Email */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">
            E-posta Adresi
          </label>
          <div className="relative">
            <Mail className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
            <input
              type="email"
              required
              value={formData.email}
              onChange={(e) => setFormData({ ...formData, email: e.target.value })}
              disabled={isLocked}
              className="w-full pl-10 pr-4 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent disabled:bg-gray-100 disabled:cursor-not-allowed"
              placeholder="admin@dansdersleri.com"
            />
          </div>
        </div>

        {/* Password */}
        <div>
          <label className="block text-sm font-medium text-gray-700 mb-2">
            Şifre
          </label>
          <div className="relative">
            <Lock className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-5 h-5" />
            <input
              type={showPassword ? 'text' : 'password'}
              required
              value={formData.password}
              onChange={(e) => setFormData({ ...formData, password: e.target.value })}
              disabled={isLocked}
              className="w-full pl-10 pr-12 py-3 border border-gray-300 rounded-lg focus:ring-2 focus:ring-primary-500 focus:border-transparent disabled:bg-gray-100 disabled:cursor-not-allowed"
              placeholder="••••••••"
            />
            <button
              type="button"
              onClick={() => setShowPassword(!showPassword)}
              className="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 hover:text-gray-600"
            >
              {showPassword ? <EyeOff className="w-5 h-5" /> : <Eye className="w-5 h-5" />}
            </button>
          </div>
        </div>

        {/* Remember Me & Forgot Password */}
        <div className="flex items-center justify-between">
          <label className="flex items-center space-x-2">
            <input
              type="checkbox"
              checked={formData.rememberMe}
              onChange={(e) => setFormData({ ...formData, rememberMe: e.target.checked })}
              className="w-4 h-4 text-primary-600 rounded focus:ring-primary-500"
            />
            <span className="text-sm text-gray-700">Beni hatırla</span>
          </label>
          <button
            type="button"
            className="text-sm text-primary-600 hover:text-primary-700 font-medium"
          >
            Şifremi Unuttum
          </button>
        </div>

        {/* Submit Button */}
        <button
          type="submit"
          disabled={loading || isLocked}
          className="w-full bg-primary-600 hover:bg-primary-700 text-white font-semibold py-3 px-6 rounded-lg transition-colors disabled:bg-gray-400 disabled:cursor-not-allowed flex items-center justify-center space-x-2"
        >
          {loading ? (
            <>
              <div className="w-5 h-5 border-2 border-white border-t-transparent rounded-full animate-spin"></div>
              <span>Giriş yapılıyor...</span>
            </>
          ) : (
            <>
              <Lock className="w-5 h-5" />
              <span>Giriş Yap</span>
            </>
          )}
        </button>
      </form>

      {/* Security Notice */}
      <div className="mt-6 pt-6 border-t border-gray-200">
        <div className="flex items-start space-x-2 text-xs text-gray-600">
          <Shield className="w-4 h-4 text-gray-400 mt-0.5" />
          <p>
            Bu sayfa SSL ile şifrelenmekte ve tüm girişler loglanmaktadır. 
            Güvenliğiniz için şifrenizi kimseyle paylaşmayın.
          </p>
        </div>
      </div>
    </div>
  )
}
