'use client'

import { useState } from 'react'
import { useRouter, useSearchParams } from 'next/navigation'
import { ChevronDown, ChevronUp } from 'lucide-react'

interface SearchParams {
  q?: string
  city?: string
  type?: string
  level?: string
  price_min?: string
  price_max?: string
  rating?: string
  page?: string
}

interface SearchFiltersProps {
  searchParams: SearchParams
}

export default function SearchFilters({ searchParams }: SearchFiltersProps) {
  const router = useRouter()
  const urlSearchParams = useSearchParams()
  const [expandedSections, setExpandedSections] = useState({
    location: true,
    danceType: true,
    level: true,
    price: true,
    rating: true
  })

  const updateFilter = (key: string, value: string | null) => {
    const params = new URLSearchParams(urlSearchParams)
    
    if (value && value !== '') {
      params.set(key, value)
    } else {
      params.delete(key)
    }
    
    params.delete('page') // Reset to first page when filters change
    router.push(`/arama?${params.toString()}`)
  }

  const toggleSection = (section: keyof typeof expandedSections) => {
    setExpandedSections(prev => ({
      ...prev,
      [section]: !prev[section]
    }))
  }

  const cities = [
    'İstanbul', 'Ankara', 'İzmir', 'Bursa', 'Antalya', 'Adana', 
    'Konya', 'Gaziantep', 'Mersin', 'Diyarbakır', 'Kayseri', 'Eskişehir'
  ]

  const danceTypes = [
    'Salsa', 'Bachata', 'Tango', 'Hip-Hop', 'Klasik Bale', 'Modern Dans',
    'Latin Dansları', 'Swing', 'Kizomba', 'Cha Cha', 'Rumba', 'Samba'
  ]

  const levels = [
    'Başlangıç', 'Orta', 'İleri', 'Tüm Seviyeler'
  ]

  const ratings = [
    { value: '4.5', label: '4.5+ Yıldız' },
    { value: '4.0', label: '4.0+ Yıldız' },
    { value: '3.5', label: '3.5+ Yıldız' },
    { value: '3.0', label: '3.0+ Yıldız' }
  ]

  return (
    <div className="bg-white rounded-xl shadow-lg p-6 space-y-6">
      <h3 className="text-lg font-semibold text-gray-900">Filtreler</h3>

      {/* Location Filter */}
      <div>
        <button
          onClick={() => toggleSection('location')}
          className="flex items-center justify-between w-full text-left font-medium text-gray-900 mb-3"
        >
          <span>Konum</span>
          {expandedSections.location ? (
            <ChevronUp className="w-4 h-4" />
          ) : (
            <ChevronDown className="w-4 h-4" />
          )}
        </button>
        
        {expandedSections.location && (
          <div className="space-y-2">
            <select
              value={searchParams.city || ''}
              onChange={(e) => updateFilter('city', e.target.value)}
              className="w-full border border-gray-300 rounded-lg px-3 py-2 text-sm focus:ring-2 focus:ring-primary-500 focus:border-transparent"
            >
              <option value="">Tüm Şehirler</option>
              {cities.map((city) => (
                <option key={city} value={city}>{city}</option>
              ))}
            </select>
          </div>
        )}
      </div>

      {/* Dance Type Filter */}
      <div>
        <button
          onClick={() => toggleSection('danceType')}
          className="flex items-center justify-between w-full text-left font-medium text-gray-900 mb-3"
        >
          <span>Dans Türü</span>
          {expandedSections.danceType ? (
            <ChevronUp className="w-4 h-4" />
          ) : (
            <ChevronDown className="w-4 h-4" />
          )}
        </button>
        
        {expandedSections.danceType && (
          <div className="space-y-2">
            <select
              value={searchParams.type || ''}
              onChange={(e) => updateFilter('type', e.target.value)}
              className="w-full border border-gray-300 rounded-lg px-3 py-2 text-sm focus:ring-2 focus:ring-primary-500 focus:border-transparent"
            >
              <option value="">Tüm Dans Türleri</option>
              {danceTypes.map((type) => (
                <option key={type} value={type}>{type}</option>
              ))}
            </select>
          </div>
        )}
      </div>

      {/* Level Filter */}
      <div>
        <button
          onClick={() => toggleSection('level')}
          className="flex items-center justify-between w-full text-left font-medium text-gray-900 mb-3"
        >
          <span>Seviye</span>
          {expandedSections.level ? (
            <ChevronUp className="w-4 h-4" />
          ) : (
            <ChevronDown className="w-4 h-4" />
          )}
        </button>
        
        {expandedSections.level && (
          <div className="space-y-2">
            {levels.map((level) => (
              <label key={level} className="flex items-center space-x-2">
                <input
                  type="radio"
                  name="level"
                  value={level}
                  checked={searchParams.level === level}
                  onChange={(e) => updateFilter('level', e.target.checked ? e.target.value : null)}
                  className="text-primary-600 focus:ring-primary-500"
                />
                <span className="text-sm text-gray-700">{level}</span>
              </label>
            ))}
          </div>
        )}
      </div>

      {/* Price Filter */}
      <div>
        <button
          onClick={() => toggleSection('price')}
          className="flex items-center justify-between w-full text-left font-medium text-gray-900 mb-3"
        >
          <span>Fiyat Aralığı</span>
          {expandedSections.price ? (
            <ChevronUp className="w-4 h-4" />
          ) : (
            <ChevronDown className="w-4 h-4" />
          )}
        </button>
        
        {expandedSections.price && (
          <div className="space-y-3">
            <div className="grid grid-cols-2 gap-2">
              <input
                type="number"
                placeholder="Min"
                value={searchParams.price_min || ''}
                onChange={(e) => updateFilter('price_min', e.target.value)}
                className="border border-gray-300 rounded-lg px-3 py-2 text-sm focus:ring-2 focus:ring-primary-500 focus:border-transparent"
              />
              <input
                type="number"
                placeholder="Max"
                value={searchParams.price_max || ''}
                onChange={(e) => updateFilter('price_max', e.target.value)}
                className="border border-gray-300 rounded-lg px-3 py-2 text-sm focus:ring-2 focus:ring-primary-500 focus:border-transparent"
              />
            </div>
            <div className="text-xs text-gray-500">
              Aylık ücret (TL)
            </div>
          </div>
        )}
      </div>

      {/* Rating Filter */}
      <div>
        <button
          onClick={() => toggleSection('rating')}
          className="flex items-center justify-between w-full text-left font-medium text-gray-900 mb-3"
        >
          <span>Minimum Puan</span>
          {expandedSections.rating ? (
            <ChevronUp className="w-4 h-4" />
          ) : (
            <ChevronDown className="w-4 h-4" />
          )}
        </button>
        
        {expandedSections.rating && (
          <div className="space-y-2">
            {ratings.map((rating) => (
              <label key={rating.value} className="flex items-center space-x-2">
                <input
                  type="radio"
                  name="rating"
                  value={rating.value}
                  checked={searchParams.rating === rating.value}
                  onChange={(e) => updateFilter('rating', e.target.checked ? e.target.value : null)}
                  className="text-primary-600 focus:ring-primary-500"
                />
                <span className="text-sm text-gray-700">{rating.label}</span>
              </label>
            ))}
          </div>
        )}
      </div>

      {/* Clear All Filters */}
      {(searchParams.city || searchParams.type || searchParams.level || searchParams.price_min || searchParams.price_max || searchParams.rating) && (
        <div className="pt-4 border-t border-gray-200">
          <button
            onClick={() => router.push('/arama')}
            className="w-full text-sm text-primary-600 hover:text-primary-700 font-medium"
          >
            Tüm Filtreleri Temizle
          </button>
        </div>
      )}
    </div>
  )
}

