# 🚀 SEO Implementation Guide - Dans Dersleri

Bu doküman, Dans Dersleri platformuna eklenen tüm SEO özelliklerini ve kullanım kılavuzunu içerir.

## 📋 İçindekiler

1. [Genel Bakış](#genel-bakış)
2. [Kurulum](#kurulum)
3. [Özellikler](#özellikler)
4. [Kullanım](#kullanım)
5. [Best Practices](#best-practices)
6. [Testing](#testing)

---

## 🎯 Genel Bakış

### Eklenen SEO Özellikleri

✅ **Technical SEO**
- Otomatik sitemap.xml
- Robots.txt
- Canonical URLs
- Meta tags (title, description, keywords)
- Structured Data (JSON-LD)
- Open Graph tags
- Twitter Cards
- Breadcrumbs

✅ **Schema Markup (10+ Tür)**
- Organization Schema
- Website Schema
- Course Schema
- LocalBusiness Schema
- FAQ Schema
- Article Schema
- Review Schema
- Video Schema
- Event Schema
- HowTo Schema
- ItemList Schema
- Breadcrumb Schema

✅ **Analytics & Tracking**
- Google Analytics 4
- Google Tag Manager
- Facebook Pixel
- Hotjar
- Custom event tracking

✅ **SEO Utilities**
- URL slug generator
- Meta description optimizer
- Image alt text generator
- Breadcrumb generator
- Keyword extractor
- Reading time calculator

---

## 🔧 Kurulum

### 1. Environment Variables

`.env.local` dosyanızı oluşturun:

```bash
# Base URL
NEXT_PUBLIC_BASE_URL=https://dansdersleri.com

# Analytics
NEXT_PUBLIC_GA_ID=G-XXXXXXXXXX
NEXT_PUBLIC_GTM_ID=GTM-XXXXXXX
NEXT_PUBLIC_FB_PIXEL_ID=123456789
NEXT_PUBLIC_HOTJAR_ID=1234567

# Webmaster Tools
NEXT_PUBLIC_GOOGLE_VERIFICATION=your-google-verification-code
NEXT_PUBLIC_BING_VERIFICATION=your-bing-verification-code
NEXT_PUBLIC_YANDEX_VERIFICATION=your-yandex-verification-code
```

### 2. Package Installation

Tüm gerekli paketler `package.json`'da mevcut:

```bash
npm install
```

---

## 📦 Özellikler

### 1. Meta Tags

Her sayfa için optimize edilmiş meta tags:

```typescript
// app/your-page/page.tsx
import { generateMetadata } from '@/lib/metadata'

export const metadata = generateCourseMetadata({
  id: '1',
  name: 'Salsa Kursu',
  description: 'Profesyonel salsa dersleri...',
  city: 'İstanbul',
  danceType: 'Salsa',
  price: 150,
  instructor: 'Maria Rodriguez',
  image: '/course-image.jpg',
})
```

### 2. Structured Data (Schema.org)

**Course Schema:**
```typescript
import { generateCourseSchema } from '@/lib/schema'
import StructuredData from '@/components/StructuredData'

const courseSchema = generateCourseSchema({
  id: '1',
  name: 'Salsa Kursu',
  description: '...',
  instructor: 'Maria Rodriguez',
  price: 150,
  currency: 'TRY',
  rating: 4.9,
  reviewCount: 127,
  image: '...',
  location: { address: '...', city: 'İstanbul' }
})

// In component
<StructuredData data={courseSchema} />
```

**FAQ Schema:**
```typescript
import { generateFAQSchema } from '@/lib/schema'

const faqSchema = generateFAQSchema([
  {
    question: 'Dans dersleri ne kadar sürer?',
    answer: 'Genellikle 6-16 hafta arasında...'
  }
])

<StructuredData data={faqSchema} />
```

**Article Schema (Blog):**
```typescript
import { generateArticleSchema } from '@/lib/schema'

const articleSchema = generateArticleSchema({
  title: 'Salsa Nasıl Öğrenilir?',
  description: '...',
  author: 'Dans Editörü',
  datePublished: '2024-01-15',
  dateModified: '2024-01-20',
  image: '/blog-image.jpg',
  url: 'https://dansdersleri.com/blog/salsa-nasil-ogrenilir'
})
```

### 3. Breadcrumbs

```typescript
import Breadcrumb from '@/components/Breadcrumb'

const breadcrumbItems = [
  { name: 'Kurslar', url: '/kurslar' },
  { name: 'İstanbul', url: '/sehirler/istanbul' },
  { name: 'Salsa Kursu', url: '/kurslar/1' }
]

<Breadcrumb items={breadcrumbItems} />
```

### 4. SEO Utilities

**Generate SEO-friendly slugs:**
```typescript
import { generateSlug } from '@/lib/seo-utils'

const slug = generateSlug('İstanbul Salsa Kursu')
// Output: 'istanbul-salsa-kursu'
```

**Extract keywords:**
```typescript
import { extractKeywords } from '@/lib/seo-utils'

const keywords = extractKeywords(longText, 10)
// Returns top 10 keywords
```

**Generate alt text:**
```typescript
import { generateImageAlt } from '@/lib/seo-utils'

const alt = generateImageAlt('Salsa Kursu', 'İstanbul Kadıköy')
// Output: 'Salsa Kursu - İstanbul Kadıköy'
```

### 5. Analytics Tracking

**Page View:**
```typescript
import { trackPageView } from '@/components/Analytics'

trackPageView('/kurslar/salsa-kursu')
```

**Custom Events:**
```typescript
import { 
  trackCourseView, 
  trackCourseClick, 
  trackSearch,
  trackContactClick 
} from '@/components/Analytics'

// Track course view
trackCourseView('1', 'Salsa Kursu')

// Track search
trackSearch('salsa istanbul', 15)

// Track contact
trackContactClick('whatsapp')
```

---

## 💡 Kullanım

### Yeni Kurs Sayfası Ekleme

1. **Metadata Oluşturma:**
```typescript
export async function generateMetadata({ params }) {
  const course = await getCourseData(params.id)
  
  return generateCourseMetadata({
    id: course.id,
    name: course.name,
    description: course.description,
    city: course.location.city,
    danceType: course.danceTypes[0],
    price: course.price,
    instructor: course.instructor.name,
    image: course.image,
  })
}
```

2. **Schema Ekleme:**
```typescript
const courseSchema = generateCourseSchema({ /* ... */ })
const breadcrumbSchema = generateBreadcrumbSchema([ /* ... */ ])
const businessSchema = generateLocalBusinessSchema({ /* ... */ })

return (
  <>
    <StructuredData data={courseSchema} />
    <StructuredData data={breadcrumbSchema} />
    <StructuredData data={businessSchema} />
    {/* Page content */}
  </>
)
```

3. **Breadcrumb Ekleme:**
```typescript
<Breadcrumb 
  items={[
    { name: 'Kurslar', url: '/kurslar' },
    { name: course.location.city, url: `/sehirler/${course.location.city}` },
    { name: course.name, url: `/kurslar/${course.id}` }
  ]} 
/>
```

### Yeni Blog Yazısı Ekleme

```typescript
// app/blog/[slug]/page.tsx
import { generateArticleSchema } from '@/lib/schema'
import { generateBlogMetadata } from '@/lib/metadata'

export async function generateMetadata({ params }) {
  const post = await getPost(params.slug)
  
  return generateBlogMetadata({
    id: post.id,
    title: post.title,
    excerpt: post.excerpt,
    author: post.author,
    publishedAt: post.publishedAt,
    updatedAt: post.updatedAt,
    image: post.image,
    tags: post.tags,
  })
}

export default async function BlogPost({ params }) {
  const post = await getPost(params.slug)
  
  const articleSchema = generateArticleSchema({
    title: post.title,
    description: post.excerpt,
    author: post.author,
    datePublished: post.publishedAt,
    dateModified: post.updatedAt,
    image: post.image,
    url: `https://dansdersleri.com/blog/${post.slug}`
  })
  
  return (
    <>
      <StructuredData data={articleSchema} />
      {/* Blog content */}
    </>
  )
}
```

---

## ✅ Best Practices

### 1. Meta Tags
- **Title**: 50-60 karakter
- **Description**: 150-160 karakter
- Her sayfada unique title ve description
- Keywords: 5-10 ana keyword

### 2. Images
- Her resme `alt` attribute ekleyin
- WebP format kullanın
- Lazy loading aktif
- Boyutları optimize edin

### 3. URLs
- SEO-friendly slugs kullanın
- Kısa ve açıklayıcı
- Türkçe karakter yerine İngilizce
- Tire (-) ile ayırın

### 4. Structured Data
- Her sayfa için uygun schema kullanın
- Test edin (Google Rich Results Test)
- Tüm gerekli alanları doldurun

### 5. Performance
- Core Web Vitals optimize edin
- LCP < 2.5s
- FID < 100ms
- CLS < 0.1

---

## 🧪 Testing

### 1. Rich Results Test
```bash
https://search.google.com/test/rich-results
```
Structured data'nızı test edin.

### 2. PageSpeed Insights
```bash
https://pagespeed.web.dev/
```
Performance ve Core Web Vitals test edin.

### 3. Schema Validator
```bash
https://validator.schema.org/
```
JSON-LD schema'larınızı validate edin.

### 4. Open Graph Debugger
```bash
https://developers.facebook.com/tools/debug/
```
Open Graph tags test edin.

### 5. Twitter Card Validator
```bash
https://cards-dev.twitter.com/validator
```
Twitter cards test edin.

---

## 📊 Monitoring

### Google Search Console
1. Site'i ekleyin
2. Sitemap gönderin: `https://dansdersleri.com/sitemap.xml`
3. Performance takip edin
4. Coverage issues kontrol edin

### Google Analytics
1. Dashboard'u kontrol edin
2. Organic traffic izleyin
3. Landing pages analiz edin
4. Conversion tracking kurun

---

## 🐛 Troubleshooting

### Sitemap görünmüyor
```bash
# Check robots.txt
curl https://dansdersleri.com/robots.txt

# Check sitemap
curl https://dansdersleri.com/sitemap.xml
```

### Schema hataları
- Schema Validator kullanın
- Required fields kontrol edin
- Date formatlarını kontrol edin (ISO 8601)

### Analytics çalışmıyor
- Environment variables kontrol edin
- Browser console'u kontrol edin
- Ad blocker devre dışı bırakın
- Network tab'ı kontrol edin

---

## 📚 Kaynaklar

- [Next.js Metadata Docs](https://nextjs.org/docs/app/building-your-application/optimizing/metadata)
- [Schema.org Documentation](https://schema.org/)
- [Google Search Central](https://developers.google.com/search)
- [Web.dev SEO Guide](https://web.dev/learn/seo/)

---

**Son Güncelleme**: Ekim 2024  
**Versiyon**: 2.0.0

