/**
 * Main JavaScript for Firma Rehberi Pro Theme
 * 
 * @package FirmaRehberiPro
 */

(function() {
    'use strict';

    // DOM ready
    document.addEventListener('DOMContentLoaded', function() {
        initMobileMenu();
        initSearch();
        initLazyLoading();
        initViewCount();
        initSmoothScrolling();
        initBackToTop();
        initFormValidation();
        initViewToggle();
        initFilters();
        initContactCountdown();
    });

    /**
     * Initialize mobile menu
     */
    function initMobileMenu() {
        const mobileToggle = document.querySelector('.mobile-menu-toggle');
        const navMenu = document.querySelector('.main-nav');
        
        if (mobileToggle && navMenu) {
            mobileToggle.addEventListener('click', function() {
                const isExpanded = this.getAttribute('aria-expanded') === 'true';
                this.setAttribute('aria-expanded', !isExpanded);
                navMenu.classList.toggle('active');
                document.body.classList.toggle('menu-open');
            });

            // Close menu when clicking outside
            document.addEventListener('click', function(e) {
                if (!mobileToggle.contains(e.target) && !navMenu.contains(e.target)) {
                    mobileToggle.setAttribute('aria-expanded', 'false');
                    navMenu.classList.remove('active');
                    document.body.classList.remove('menu-open');
                }
            });

            // Close menu on escape key
            document.addEventListener('keydown', function(e) {
                if (e.key === 'Escape') {
                    mobileToggle.setAttribute('aria-expanded', 'false');
                    navMenu.classList.remove('active');
                    document.body.classList.remove('menu-open');
                }
            });
        }
    }

    /**
     * Initialize search functionality
     */
    function initSearch() {
        const searchForm = document.querySelector('.search-form');
        const searchInput = document.querySelector('.search-input');
        
        if (searchForm && searchInput) {
            // Add search suggestions
            searchInput.addEventListener('input', debounce(function() {
                const query = this.value.trim();
                if (query.length >= 2) {
                    showSearchSuggestions(query);
                } else {
                    hideSearchSuggestions();
                }
            }, 300));

            // Handle form submission
            searchForm.addEventListener('submit', function(e) {
                const query = searchInput.value.trim();
                if (!query) {
                    e.preventDefault();
                    searchInput.focus();
                    return;
                }
            });

            // Hide suggestions when clicking outside
            document.addEventListener('click', function(e) {
                if (!searchForm.contains(e.target)) {
                    hideSearchSuggestions();
                }
            });
        }
    }

    /**
     * Show search suggestions
     */
    function showSearchSuggestions(query) {
        if (typeof firmaRehberi === 'undefined') return;

        fetch(firmaRehberi.ajaxUrl, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: new URLSearchParams({
                action: 'get_search_suggestions',
                query: query,
                nonce: firmaRehberi.nonce
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                displaySearchSuggestions(data.data);
            }
        })
        .catch(error => {
            console.error('Search suggestions error:', error);
        });
    }

    /**
     * Display search suggestions
     */
    function displaySearchSuggestions(suggestions) {
        const searchForm = document.querySelector('.search-form');
        if (!searchForm) return;

        // Remove existing suggestions
        hideSearchSuggestions();

        if (suggestions.length === 0) return;

        const suggestionsContainer = document.createElement('div');
        suggestionsContainer.className = 'search-suggestions';
        suggestionsContainer.innerHTML = suggestions.map(item => `
            <div class="suggestion-item" data-url="${item.url}">
                <div class="suggestion-title">${item.title}</div>
                <div class="suggestion-description">${item.description || ''}</div>
            </div>
        `).join('');

        // Add click handlers
        suggestionsContainer.addEventListener('click', function(e) {
            const item = e.target.closest('.suggestion-item');
            if (item) {
                window.location.href = item.dataset.url;
            }
        });

        searchForm.appendChild(suggestionsContainer);
    }

    /**
     * Hide search suggestions
     */
    function hideSearchSuggestions() {
        const suggestions = document.querySelector('.search-suggestions');
        if (suggestions) {
            suggestions.remove();
        }
    }

    /**
     * Initialize lazy loading
     */
    function initLazyLoading() {
        if ('IntersectionObserver' in window) {
            const imageObserver = new IntersectionObserver((entries, observer) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        const img = entry.target;
                        if (img.dataset.src) {
                            img.src = img.dataset.src;
                            img.classList.remove('lazy');
                            observer.unobserve(img);
                        }
                    }
                });
            });

            document.querySelectorAll('img[data-src]').forEach(img => {
                imageObserver.observe(img);
            });
        }
    }

    /**
     * Initialize view count tracking
     */
    function initViewCount() {
        if (typeof firmaRehberi === 'undefined') return;

        // Track company page views
        if (document.body.classList.contains('single-company')) {
            const postId = document.querySelector('body').getAttribute('data-post-id');
            if (postId) {
                trackViewCount(postId);
            }
        }

        // Track company card views
        document.addEventListener('click', function(e) {
            const companyCard = e.target.closest('.company-card');
            if (companyCard) {
                const postId = companyCard.getAttribute('data-post-id');
                if (postId) {
                    trackViewCount(postId);
                }
            }
        });
    }

    /**
     * Track view count
     */
    function trackViewCount(postId) {
        if (typeof firmaRehberi === 'undefined') return;

        fetch(firmaRehberi.ajaxUrl, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/x-www-form-urlencoded',
            },
            body: new URLSearchParams({
                action: 'increment_view_count',
                post_id: postId,
                nonce: firmaRehberi.nonce
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Update view count display
                const viewElement = document.querySelector(`[data-post-id="${postId}"] .view-count`);
                if (viewElement) {
                    viewElement.textContent = data.data.views;
                }
            }
        })
        .catch(error => {
            console.error('View count tracking error:', error);
        });
    }

    /**
     * Initialize smooth scrolling
     */
    function initSmoothScrolling() {
        document.querySelectorAll('a[href^="#"]').forEach(link => {
            link.addEventListener('click', function(e) {
                const targetId = this.getAttribute('href').substring(1);
                const targetElement = document.getElementById(targetId);
                
                if (targetElement) {
                    e.preventDefault();
                    targetElement.scrollIntoView({
                        behavior: 'smooth',
                        block: 'start'
                    });
                }
            });
        });
    }

    /**
     * Initialize back to top button
     */
    function initBackToTop() {
        const backToTopButton = document.createElement('button');
        backToTopButton.className = 'back-to-top';
        backToTopButton.innerHTML = '↑';
        backToTopButton.setAttribute('aria-label', 'Yukarı çık');
        backToTopButton.style.cssText = `
            position: fixed;
            bottom: 20px;
            right: 20px;
            width: 50px;
            height: 50px;
            border-radius: 50%;
            background: #1e40af;
            color: white;
            border: none;
            font-size: 20px;
            cursor: pointer;
            display: none;
            z-index: 1000;
            transition: all 0.3s;
            box-shadow: 0 4px 15px rgba(30, 64, 175, 0.3);
        `;
        
        document.body.appendChild(backToTopButton);
        
        // Show/hide button based on scroll position
        window.addEventListener('scroll', function() {
            if (window.pageYOffset > 300) {
                backToTopButton.style.display = 'block';
            } else {
                backToTopButton.style.display = 'none';
            }
        });
        
        // Scroll to top when clicked
        backToTopButton.addEventListener('click', function() {
            window.scrollTo({
                top: 0,
                behavior: 'smooth'
            });
        });
    }

    /**
     * Initialize form validation
     */
    function initFormValidation() {
        const forms = document.querySelectorAll('form');
        
        forms.forEach(form => {
            form.addEventListener('submit', function(e) {
                const requiredFields = this.querySelectorAll('[required]');
                let isValid = true;
                
                requiredFields.forEach(field => {
                    if (!field.value.trim()) {
                        isValid = false;
                        field.classList.add('error');
                        showFieldError(field, 'Bu alan zorunludur');
                    } else {
                        field.classList.remove('error');
                        hideFieldError(field);
                    }
                });
                
                if (!isValid) {
                    e.preventDefault();
                    showNotification('Lütfen tüm gerekli alanları doldurun', 'error');
                }
            });
        });
    }

    /**
     * Show field error
     */
    function showFieldError(field, message) {
        hideFieldError(field);
        
        const errorElement = document.createElement('div');
        errorElement.className = 'field-error';
        errorElement.textContent = message;
        errorElement.style.cssText = 'color: #e74c3c; font-size: 0.875rem; margin-top: 0.25rem;';
        
        field.parentNode.appendChild(errorElement);
    }

    /**
     * Hide field error
     */
    function hideFieldError(field) {
        const errorElement = field.parentNode.querySelector('.field-error');
        if (errorElement) {
            errorElement.remove();
        }
    }

    /**
     * Initialize view toggle
     */
    function initViewToggle() {
        const viewToggles = document.querySelectorAll('.view-toggle');
        const companiesGrid = document.querySelector('.companies-grid');
        
        if (viewToggles.length && companiesGrid) {
            viewToggles.forEach(toggle => {
                toggle.addEventListener('click', function() {
                    const view = this.dataset.view;
                    
                    // Update active state
                    viewToggles.forEach(t => t.classList.remove('active'));
                    this.classList.add('active');
                    
                    // Update grid class
                    companiesGrid.className = 'companies-grid ' + view + '-view';
                    
                    // Store preference
                    localStorage.setItem('company-view', view);
                });
            });
            
            // Restore view preference
            const savedView = localStorage.getItem('company-view');
            if (savedView) {
                const toggle = document.querySelector(`[data-view="${savedView}"]`);
                if (toggle) {
                    toggle.click();
                }
            }
        }
    }

    /**
     * Initialize filters
     */
    function initFilters() {
        const filterSelects = document.querySelectorAll('.filter-select');
        
        filterSelects.forEach(select => {
            select.addEventListener('change', function() {
                // Add loading state
                const form = this.closest('form');
                if (form) {
                    form.classList.add('loading');
                }
                
                // Submit form
                this.form.submit();
            });
        });
    }

    /**
     * Show notification
     */
    function showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.textContent = message;
        notification.style.cssText = `
            position: fixed;
            top: 20px;
            right: 20px;
            padding: 1rem 1.5rem;
            border-radius: 6px;
            color: white;
            font-weight: 600;
            z-index: 10000;
            transform: translateX(100%);
            transition: transform 0.3s ease;
            background: ${type === 'error' ? '#e74c3c' : '#1e40af'};
        `;
        
        document.body.appendChild(notification);
        
        // Animate in
        setTimeout(() => {
            notification.style.transform = 'translateX(0)';
        }, 100);
        
        // Remove after 3 seconds
        setTimeout(() => {
            notification.style.transform = 'translateX(100%)';
            setTimeout(() => {
                notification.remove();
            }, 300);
        }, 3000);
    }

    /**
     * Debounce function
     */
    function debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }

    /**
     * Throttle function
     */
    function throttle(func, limit) {
        let inThrottle;
        return function() {
            const args = arguments;
            const context = this;
            if (!inThrottle) {
                func.apply(context, args);
                inThrottle = true;
                setTimeout(() => inThrottle = false, limit);
            }
        };
    }

    /**
     * Initialize 30 second contact countdown
     */
    function initContactCountdown() {
        const contactInfo = document.querySelector('.contact-info');
        const countdownElement = document.getElementById('contact-countdown');
        const countdownTime = document.getElementById('countdown-time');
        
        if (!contactInfo || !countdownElement || !countdownTime) return;
        
        let timeLeft = 30;
        let isPaused = false;
        
        // İletişim bilgilerini gizle
        contactInfo.style.display = 'none';
        
        // Geri sayım fonksiyonu
        function updateCountdown() {
            if (isPaused) return;
            
            countdownTime.textContent = `Kalan süre: ${timeLeft} saniye`;
            
            if (timeLeft <= 0) {
                // Geri sayım bitti
                countdownElement.style.display = 'none';
                contactInfo.style.display = 'block';
                contactInfo.style.animation = 'fadeInUp 0.5s ease-in-out';
                
                // Başarı mesajı
                showSuccessMessage();
            } else {
                timeLeft--;
                setTimeout(updateCountdown, 1000);
            }
        }
        
        // Başarı mesajı
        function showSuccessMessage() {
            const successDiv = document.createElement('div');
            successDiv.className = 'success-message bg-green-100 border border-green-400 text-green-700 px-4 py-3 rounded mb-4';
            successDiv.innerHTML = '✅ İletişim bilgileri başarıyla yüklendi!';
            
            contactInfo.insertBefore(successDiv, contactInfo.firstChild);
            
            // 3 saniye sonra mesajı kaldır
            setTimeout(() => {
                successDiv.remove();
            }, 3000);
        }
        
        // Geri sayımı başlat
        updateCountdown();
        
        // Sayfa görünürlüğü değiştiğinde duraklat/devam et
        document.addEventListener('visibilitychange', function() {
            isPaused = document.hidden;
        });
    }

    // Add CSS for search suggestions
    const style = document.createElement('style');
    style.textContent = `
        .search-suggestions {
            position: absolute;
            top: 100%;
            left: 0;
            right: 0;
            background: white;
            border: 1px solid #e5e7eb;
            border-radius: 0 0 8px 8px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
            z-index: 1000;
            max-height: 300px;
            overflow-y: auto;
        }
        
        .suggestion-item {
            padding: 0.75rem 1rem;
            cursor: pointer;
            border-bottom: 1px solid #f3f4f6;
            transition: background-color 0.2s;
        }
        
        .suggestion-item:hover {
            background-color: #f8fafc;
        }
        
        .suggestion-item:last-child {
            border-bottom: none;
        }
        
        .suggestion-title {
            font-weight: 600;
            color: #1a1a1a;
            margin-bottom: 0.25rem;
        }
        
        .suggestion-description {
            font-size: 0.875rem;
            color: #6b7280;
        }
        
        .loading {
            opacity: 0.6;
            pointer-events: none;
        }
        
        .loading::after {
            content: '';
            position: absolute;
            top: 50%;
            left: 50%;
            width: 20px;
            height: 20px;
            margin: -10px 0 0 -10px;
            border: 2px solid #f3f3f3;
            border-top: 2px solid #1e40af;
            border-radius: 50%;
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        .field-error {
            color: #e74c3c;
            font-size: 0.875rem;
            margin-top: 0.25rem;
        }
        
        .error {
            border-color: #e74c3c !important;
            box-shadow: 0 0 0 3px rgba(231, 76, 60, 0.1) !important;
        }
    `;
    document.head.appendChild(style);

})();
