<?php
/**
 * Firma Rehberi Pro Theme Functions
 * 
 * @package FirmaRehberiPro
 * @version 1.0.0
 * Requires at least: 6.6
 * Requires PHP: 8.1
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define theme constants
define('FIRMA_REHBERI_VERSION', '1.0.0');
define('FIRMA_REHBERI_THEME_DIR', get_template_directory());
define('FIRMA_REHBERI_THEME_URL', get_template_directory_uri());

/**
 * Theme Setup
 */
function firma_rehberi_setup() {
    // Add theme support
    add_theme_support('post-thumbnails');
    add_theme_support('title-tag');
    add_theme_support('custom-logo', array(
        'height'      => 100,
        'width'       => 300,
        'flex-height' => true,
        'flex-width'  => true,
    ));
    add_theme_support('html5', array(
        'search-form',
        'comment-form',
        'comment-list',
        'gallery',
        'caption',
        'style',
        'script',
    ));
    add_theme_support('customize-selective-refresh-widgets');
    add_theme_support('wp-block-styles');
    add_theme_support('align-wide');
    add_theme_support('editor-styles');
    add_theme_support('responsive-embeds');
    
    // Add image sizes
    add_image_size('company-logo', 300, 300, true);
    add_image_size('company-hero', 800, 400, true);
    add_image_size('company-thumbnail', 150, 150, true);
    
    // Register navigation menus
    register_nav_menus(array(
        'primary' => __('Ana Menü', 'firma-rehberi-pro'),
        'footer'  => __('Footer Menü', 'firma-rehberi-pro'),
    ));
    
    // Load theme textdomain
    load_theme_textdomain('firma-rehberi-pro', get_template_directory() . '/languages');
}
add_action('after_setup_theme', 'firma_rehberi_setup');

/**
 * Enqueue scripts and styles
 */
function firma_rehberi_enqueue_scripts() {
    // CSS
    wp_enqueue_style(
        'firma-rehberi-style',
        get_template_directory_uri() . '/style.css',
        array(),
        '1.0.0'
    );
    
    // JavaScript
    wp_enqueue_script(
        'firma-rehberi-main',
        get_template_directory_uri() . '/assets/js/main.js',
        array(),
        '1.0.0',
        true
    );
    
    // Localize script for AJAX
    wp_localize_script('firma-rehberi-main', 'firmaRehberi', array(
        'ajaxUrl' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('firma_rehberi_nonce'),
    ));
}
add_action('wp_enqueue_scripts', 'firma_rehberi_enqueue_scripts');

/**
 * Include required files
 */
require_once FIRMA_REHBERI_THEME_DIR . '/inc/php-compatibility.php';
require_once FIRMA_REHBERI_THEME_DIR . '/inc/performance-php81.php';
require_once FIRMA_REHBERI_THEME_DIR . '/inc/custom-post-type.php';
require_once FIRMA_REHBERI_THEME_DIR . '/inc/schema-generator.php';
require_once FIRMA_REHBERI_THEME_DIR . '/inc/seo-functions.php';
require_once FIRMA_REHBERI_THEME_DIR . '/inc/enqueue.php';
require_once FIRMA_REHBERI_THEME_DIR . '/inc/helpers.php';

/**
 * Register widget areas
 */
function firma_rehberi_widgets_init() {
    register_sidebar(array(
        'name'          => __('Ana Sidebar', 'firma-rehberi-pro'),
        'id'            => 'sidebar-1',
        'description'   => __('Ana sayfa sidebar alanı', 'firma-rehberi-pro'),
        'before_widget' => '<section id="%1$s" class="widget %2$s">',
        'after_widget'  => '</section>',
        'before_title'  => '<h3 class="widget-title">',
        'after_title'   => '</h3>',
    ));
    
    register_sidebar(array(
        'name'          => __('Footer 1', 'firma-rehberi-pro'),
        'id'            => 'footer-1',
        'description'   => __('Footer 1. sütun', 'firma-rehberi-pro'),
        'before_widget' => '<div id="%1$s" class="footer-widget %2$s">',
        'after_widget'  => '</div>',
        'before_title'  => '<h3 class="footer-widget-title">',
        'after_title'   => '</h3>',
    ));
    
    register_sidebar(array(
        'name'          => __('Footer 2', 'firma-rehberi-pro'),
        'id'            => 'footer-2',
        'description'   => __('Footer 2. sütun', 'firma-rehberi-pro'),
        'before_widget' => '<div id="%1$s" class="footer-widget %2$s">',
        'after_widget'  => '</div>',
        'before_title'  => '<h3 class="footer-widget-title">',
        'after_title'   => '</h3>',
    ));
    
    register_sidebar(array(
        'name'          => __('Footer 3', 'firma-rehberi-pro'),
        'id'            => 'footer-3',
        'description'   => __('Footer 3. sütun', 'firma-rehberi-pro'),
        'before_widget' => '<div id="%1$s" class="footer-widget %2$s">',
        'after_widget'  => '</div>',
        'before_title'  => '<h3 class="footer-widget-title">',
        'after_title'   => '</h3>',
    ));
}
add_action('widgets_init', 'firma_rehberi_widgets_init');

/**
 * Custom excerpt length
 */
function firma_rehberi_excerpt_length($length) {
    return 20;
}
add_filter('excerpt_length', 'firma_rehberi_excerpt_length');

/**
 * Custom excerpt more
 */
function firma_rehberi_excerpt_more($more) {
    return '...';
}
add_filter('excerpt_more', 'firma_rehberi_excerpt_more');

/**
 * Add custom body classes
 */
function firma_rehberi_body_classes($classes) {
    if (is_front_page()) {
        $classes[] = 'home-page';
    }
    
    if (is_singular('company')) {
        $classes[] = 'single-company';
    }
    
    if (is_post_type_archive('company')) {
        $classes[] = 'company-archive';
    }
    
    return $classes;
}
add_filter('body_class', 'firma_rehberi_body_classes');

/**
 * Add custom post type to main query
 */
function firma_rehberi_add_company_to_main_query($query) {
    if (!is_admin() && $query->is_main_query()) {
        if (is_home() || is_category() || is_tag()) {
            $query->set('post_type', array('post', 'company'));
        }
    }
}
add_action('pre_get_posts', 'firma_rehberi_add_company_to_main_query');

/**
 * Custom search query
 */
function firma_rehberi_search_query($query) {
    if (!is_admin() && $query->is_main_query()) {
        if (is_search()) {
            $query->set('post_type', array('post', 'company'));
        }
    }
}
add_action('pre_get_posts', 'firma_rehberi_search_query');

/**
 * Add view count functionality
 */
function firma_rehberi_increment_view_count() {
    if (is_singular('company') && !is_admin()) {
        global $post;
        $views = get_post_meta($post->ID, '_view_count', true);
        $views = $views ? $views + 1 : 1;
        update_post_meta($post->ID, '_view_count', $views);
    }
}
add_action('wp_head', 'firma_rehberi_increment_view_count');

/**
 * AJAX view count increment
 */
function firma_rehberi_ajax_increment_view_count() {
    check_ajax_referer('firma_rehberi_nonce', 'nonce');
    
    $post_id = intval($_POST['post_id']);
    if ($post_id) {
        $views = get_post_meta($post_id, '_view_count', true);
        $views = $views ? $views + 1 : 1;
        update_post_meta($post_id, '_view_count', $views);
        wp_send_json_success(array('views' => $views));
    }
    
    wp_send_json_error();
}
add_action('wp_ajax_increment_view_count', 'firma_rehberi_ajax_increment_view_count');
add_action('wp_ajax_nopriv_increment_view_count', 'firma_rehberi_ajax_increment_view_count');

/**
 * Add schema.org structured data
 */
function firma_rehberi_add_schema_markup() {
    if (is_singular('company')) {
        $schema = firma_rehberi_generate_company_schema();
        if ($schema) {
            echo '<script type="application/ld+json">' . wp_json_encode($schema, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE) . '</script>';
        }
    }
    
    if (is_post_type_archive('company')) {
        $schema = firma_rehberi_generate_archive_schema();
        if ($schema) {
            echo '<script type="application/ld+json">' . wp_json_encode($schema, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE) . '</script>';
        }
    }
    
    if (is_home() || is_front_page()) {
        $schema = firma_rehberi_generate_website_schema();
        if ($schema) {
            echo '<script type="application/ld+json">' . wp_json_encode($schema, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE) . '</script>';
        }
    }
}
add_action('wp_head', 'firma_rehberi_add_schema_markup');

/**
 * Add breadcrumb schema
 */
function firma_rehberi_add_breadcrumb_schema() {
    if (!is_front_page()) {
        $breadcrumb_schema = firma_rehberi_generate_breadcrumb_schema();
        if ($breadcrumb_schema) {
            echo '<script type="application/ld+json">' . wp_json_encode($breadcrumb_schema, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE) . '</script>';
        }
    }
}
add_action('wp_head', 'firma_rehberi_add_breadcrumb_schema');

/**
 * Add Open Graph and Twitter Card meta tags
 */
function firma_rehberi_add_og_meta_tags() {
    if (is_singular('company')) {
        global $post;
        $company_name = get_the_title();
        $description = get_the_excerpt() ?: get_bloginfo('description');
        $image = get_the_post_thumbnail_url($post->ID, 'large') ?: get_template_directory_uri() . '/assets/images/default-og-image.jpg';
        
        echo '<meta property="og:title" content="' . esc_attr($company_name) . '">' . "\n";
        echo '<meta property="og:description" content="' . esc_attr($description) . '">' . "\n";
        echo '<meta property="og:image" content="' . esc_url($image) . '">' . "\n";
        echo '<meta property="og:url" content="' . esc_url(get_permalink()) . '">' . "\n";
        echo '<meta property="og:type" content="business.business">' . "\n";
        echo '<meta property="og:site_name" content="' . esc_attr(get_bloginfo('name')) . '">' . "\n";
        
        echo '<meta name="twitter:card" content="summary_large_image">' . "\n";
        echo '<meta name="twitter:title" content="' . esc_attr($company_name) . '">' . "\n";
        echo '<meta name="twitter:description" content="' . esc_attr($description) . '">' . "\n";
        echo '<meta name="twitter:image" content="' . esc_url($image) . '">' . "\n";
    }
}
add_action('wp_head', 'firma_rehberi_add_og_meta_tags');

/**
 * Add canonical URL
 */
function firma_rehberi_add_canonical_url() {
    if (is_singular()) {
        echo '<link rel="canonical" href="' . esc_url(get_permalink()) . '">' . "\n";
    } elseif (is_post_type_archive()) {
        echo '<link rel="canonical" href="' . esc_url(get_post_type_archive_link(get_post_type())) . '">' . "\n";
    }
}
add_action('wp_head', 'firma_rehberi_add_canonical_url');

/**
 * Add robots meta tag
 */
function firma_rehberi_add_robots_meta() {
    if (is_singular('company')) {
        echo '<meta name="robots" content="index, follow, max-snippet:-1, max-image-preview:large, max-video-preview:-1">' . "\n";
    } else {
        echo '<meta name="robots" content="index, follow">' . "\n";
    }
}
add_action('wp_head', 'firma_rehberi_add_robots_meta');

/**
 * Add viewport meta tag
 */
function firma_rehberi_add_viewport_meta() {
    echo '<meta name="viewport" content="width=device-width, initial-scale=1.0">' . "\n";
}
add_action('wp_head', 'firma_rehberi_add_viewport_meta', 1);

/**
 * Add preconnect for performance
 */
function firma_rehberi_add_preconnect_links() {
    echo '<link rel="preconnect" href="https://fonts.googleapis.com">' . "\n";
    echo '<link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>' . "\n";
}
add_action('wp_head', 'firma_rehberi_add_preconnect_links', 1);

/**
 * Add skip link for accessibility
 */
function firma_rehberi_add_skip_link() {
    echo '<a class="skip-link" href="#main">Ana içeriğe geç</a>' . "\n";
}
add_action('wp_body_open', 'firma_rehberi_add_skip_link');

/**
 * Customize admin bar
 */
function firma_rehberi_customize_admin_bar($wp_admin_bar) {
    if (current_user_can('edit_posts')) {
        $wp_admin_bar->add_node(array(
            'id'    => 'add-company',
            'title' => 'Yeni Firma Ekle',
            'href'  => admin_url('post-new.php?post_type=company'),
        ));
    }
}
add_action('admin_bar_menu', 'firma_rehberi_customize_admin_bar', 999);

/**
 * Add customizer options
 */
function firma_rehberi_customize_register($wp_customize) {
    // Add Google Maps API section
    $wp_customize->add_section('firma_rehberi_maps', array(
        'title'    => __('Google Maps Ayarları', 'firma-rehberi-pro'),
        'priority' => 30,
    ));
    
    $wp_customize->add_setting('google_maps_api_key', array(
        'default'           => '',
        'sanitize_callback' => 'sanitize_text_field',
    ));
    
    $wp_customize->add_control('google_maps_api_key', array(
        'label'   => __('Google Maps API Key', 'firma-rehberi-pro'),
        'section' => 'firma_rehberi_maps',
        'type'    => 'text',
    ));
    
    // Add SEO settings section
    $wp_customize->add_section('firma_rehberi_seo', array(
        'title'    => __('SEO Ayarları', 'firma-rehberi-pro'),
        'priority' => 35,
    ));
    
    $wp_customize->add_setting('default_meta_description', array(
        'default'           => '',
        'sanitize_callback' => 'sanitize_textarea_field',
    ));
    
    $wp_customize->add_control('default_meta_description', array(
        'label'   => __('Varsayılan Meta Açıklama', 'firma-rehberi-pro'),
        'section' => 'firma_rehberi_seo',
        'type'    => 'textarea',
    ));
}
add_action('customize_register', 'firma_rehberi_customize_register');

/**
 * Flush rewrite rules on theme activation
 */
function firma_rehberi_flush_rewrite_rules() {
    firma_rehberi_register_company_post_type();
    flush_rewrite_rules();
}
register_activation_hook(__FILE__, 'firma_rehberi_flush_rewrite_rules');

/**
 * Add theme support for Gutenberg
 */
function firma_rehberi_gutenberg_support() {
    // Add theme support for editor styles
    add_theme_support('editor-styles');
    
    // Add custom editor styles
    add_editor_style('assets/css/editor-style.css');
}
add_action('after_setup_theme', 'firma_rehberi_gutenberg_support');

/**
 * Add custom CSS for admin
 */
function firma_rehberi_admin_styles() {
    echo '<style>
        .post-type-company .wp-list-table .column-title { width: 30%; }
        .post-type-company .wp-list-table .column-company_category { width: 20%; }
        .post-type-company .wp-list-table .column-company_city { width: 15%; }
        .post-type-company .wp-list-table .column-view_count { width: 10%; }
    </style>';
}
add_action('admin_head', 'firma_rehberi_admin_styles');

/**
 * Add custom columns to company admin list
 */
function firma_rehberi_add_company_columns($columns) {
    $new_columns = array();
    $new_columns['cb'] = $columns['cb'];
    $new_columns['title'] = $columns['title'];
    $new_columns['company_category'] = __('Kategori', 'firma-rehberi-pro');
    $new_columns['company_city'] = __('Şehir', 'firma-rehberi-pro');
    $new_columns['view_count'] = __('Görüntülenme', 'firma-rehberi-pro');
    $new_columns['date'] = $columns['date'];
    
    return $new_columns;
}
add_filter('manage_company_posts_columns', 'firma_rehberi_add_company_columns');

/**
 * Populate custom columns
 */
function firma_rehberi_populate_company_columns($column, $post_id) {
    switch ($column) {
        case 'company_category':
            $terms = get_the_terms($post_id, 'company_category');
            if ($terms && !is_wp_error($terms)) {
                echo esc_html($terms[0]->name);
            }
            break;
            
        case 'company_city':
            $terms = get_the_terms($post_id, 'company_city');
            if ($terms && !is_wp_error($terms)) {
                echo esc_html($terms[0]->name);
            }
            break;
            
        case 'view_count':
            $views = get_post_meta($post_id, '_view_count', true);
            echo esc_html($views ?: 0);
            break;
    }
}
add_action('manage_company_posts_custom_column', 'firma_rehberi_populate_company_columns', 10, 2);

/**
 * Make custom columns sortable
 */
function firma_rehberi_sortable_company_columns($columns) {
    $columns['view_count'] = 'view_count';
    return $columns;
}
add_filter('manage_edit-company_sortable_columns', 'firma_rehberi_sortable_company_columns');

/**
 * Handle custom column sorting
 */
function firma_rehberi_handle_company_column_sorting($query) {
    if (!is_admin() || !$query->is_main_query()) {
        return;
    }
    
    $orderby = $query->get('orderby');
    
    if ('view_count' === $orderby) {
        $query->set('meta_key', '_view_count');
        $query->set('orderby', 'meta_value_num');
    }
}
add_action('pre_get_posts', 'firma_rehberi_handle_company_column_sorting');
