<?php
/**
 * Template part for displaying company cards
 *
 * @package FirmaRehberiPro
 */

$company_id = get_the_ID();
$company_name = get_the_title();
$company_excerpt = get_the_excerpt();
$company_logo = get_the_post_thumbnail_url($company_id, 'company-thumbnail');
$company_rating = get_post_meta($company_id, '_rating', true);
$company_phone = get_post_meta($company_id, '_phone', true);
$company_address = get_post_meta($company_id, '_address', true);
$view_count = get_post_meta($company_id, '_view_count', true) ?: 0;

// Get categories
$categories = get_the_terms($company_id, 'company_category');
$category_name = $categories && !is_wp_error($categories) ? $categories[0]->name : '';

// Get cities
$cities = get_the_terms($company_id, 'company_city');
$city_name = $cities && !is_wp_error($cities) ? $cities[0]->name : '';
?>

<article class="company-card" data-post-id="<?php echo esc_attr($company_id); ?>">
    <div class="company-logo-container">
        <?php if ($company_logo) : ?>
            <img src="<?php echo esc_url($company_logo); ?>" alt="<?php echo esc_attr($company_name); ?>" class="company-logo" loading="lazy">
        <?php else : ?>
            <div class="company-logo-placeholder">
                <span class="logo-text"><?php echo esc_html(substr($company_name, 0, 2)); ?></span>
            </div>
        <?php endif; ?>
    </div>
    
    <div class="company-content">
        <header class="company-header">
            <h3 class="company-name">
                <a href="<?php the_permalink(); ?>" rel="bookmark"><?php echo esc_html($company_name); ?></a>
            </h3>
            
            <?php if ($category_name) : ?>
                <div class="company-category"><?php echo esc_html($category_name); ?></div>
            <?php endif; ?>
            
            <?php if ($city_name) : ?>
                <div class="company-location">📍 <?php echo esc_html($city_name); ?></div>
            <?php endif; ?>
        </header>
        
        <?php if ($company_rating) : ?>
            <div class="company-rating">
                <div class="stars">
                    <?php for ($i = 1; $i <= 5; $i++) : ?>
                        <span class="star <?php echo $i <= $company_rating ? 'filled' : ''; ?>">★</span>
                    <?php endfor; ?>
                </div>
                <span class="rating-text"><?php echo esc_html($company_rating); ?>/5</span>
            </div>
        <?php endif; ?>
        
        <?php if ($company_excerpt) : ?>
            <div class="company-description">
                <?php echo wp_trim_words($company_excerpt, 20); ?>
            </div>
        <?php endif; ?>
        
        <div class="company-meta">
            <?php if ($company_phone) : ?>
                <div class="meta-item">
                    <span class="meta-icon">📞</span>
                    <a href="tel:<?php echo esc_attr($company_phone); ?>"><?php echo esc_html($company_phone); ?></a>
                </div>
            <?php endif; ?>
            
            <?php if ($company_address) : ?>
                <div class="meta-item">
                    <span class="meta-icon">📍</span>
                    <span><?php echo esc_html(wp_trim_words($company_address, 8)); ?></span>
                </div>
            <?php endif; ?>
            
            <div class="meta-item">
                <span class="meta-icon">👁️</span>
                <span><?php echo esc_html($view_count); ?> <?php _e('görüntülenme', 'firma-rehberi-pro'); ?></span>
            </div>
        </div>
        
        <footer class="company-footer">
            <a href="<?php the_permalink(); ?>" class="view-details-btn">
                <?php _e('Detayları Gör', 'firma-rehberi-pro'); ?>
            </a>
            
            <?php if ($company_phone) : ?>
                <a href="tel:<?php echo esc_attr($company_phone); ?>" class="contact-btn">
                    <?php _e('Ara', 'firma-rehberi-pro'); ?>
                </a>
            <?php endif; ?>
        </footer>
    </div>
</article>

<style>
.company-card {
    background: #ffffff;
    border-radius: 12px;
    padding: 1.5rem;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
    transition: all 0.2s ease;
    border: 1px solid #e5e7eb;
    position: relative;
    overflow: hidden;
}

.company-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1);
}

.company-card::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 4px;
    background: linear-gradient(90deg, #1e40af, #3b82f6);
}

.company-logo-container {
    text-align: center;
    margin-bottom: 1rem;
}

.company-logo {
    width: 80px;
    height: 80px;
    border-radius: 8px;
    object-fit: cover;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
}

.company-logo-placeholder {
    width: 80px;
    height: 80px;
    background: linear-gradient(135deg, #1e40af, #3b82f6);
    color: #ffffff;
    border-radius: 8px;
    display: inline-flex;
    align-items: center;
    justify-content: center;
    font-size: 1.5rem;
    font-weight: 700;
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
}

.company-content {
    text-align: center;
}

.company-header {
    margin-bottom: 1rem;
}

.company-name {
    font-size: 1.25rem;
    font-weight: 700;
    margin-bottom: 0.5rem;
    line-height: 1.3;
}

.company-name a {
    color: #1a1a1a;
    text-decoration: none;
}

.company-name a:hover {
    color: #1e40af;
}

.company-category {
    background: #f1f5f9;
    color: #1e40af;
    padding: 0.25rem 0.75rem;
    border-radius: 12px;
    font-size: 0.75rem;
    font-weight: 600;
    display: inline-block;
    margin-bottom: 0.5rem;
}

.company-location {
    color: #6b7280;
    font-size: 0.875rem;
    margin-bottom: 1rem;
}

.company-rating {
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 0.5rem;
    margin-bottom: 1rem;
}

.stars {
    display: flex;
    gap: 0.125rem;
}

.star {
    color: #d1d5db;
    font-size: 1rem;
}

.star.filled {
    color: #fbbf24;
}

.rating-text {
    color: #6b7280;
    font-size: 0.875rem;
    font-weight: 500;
}

.company-description {
    color: #6b7280;
    font-size: 0.875rem;
    line-height: 1.5;
    margin-bottom: 1rem;
    text-align: left;
}

.company-meta {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
    margin-bottom: 1.5rem;
    text-align: left;
}

.meta-item {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    font-size: 0.875rem;
    color: #6b7280;
}

.meta-icon {
    width: 16px;
    height: 16px;
    text-align: center;
    flex-shrink: 0;
}

.meta-item a {
    color: #1e40af;
    text-decoration: none;
}

.meta-item a:hover {
    text-decoration: underline;
}

.company-footer {
    display: flex;
    gap: 0.75rem;
    justify-content: center;
    flex-wrap: wrap;
}

.view-details-btn,
.contact-btn {
    padding: 0.75rem 1.5rem;
    border-radius: 6px;
    text-decoration: none;
    font-weight: 600;
    font-size: 0.875rem;
    transition: all 0.2s ease;
    flex: 1;
    text-align: center;
    min-width: 120px;
}

.view-details-btn {
    background: #1e40af;
    color: #ffffff;
}

.view-details-btn:hover {
    background: #1d4ed8;
    color: #ffffff;
    text-decoration: none;
}

.contact-btn {
    background: #ffffff;
    color: #1e40af;
    border: 1px solid #1e40af;
}

.contact-btn:hover {
    background: #1e40af;
    color: #ffffff;
    text-decoration: none;
}

/* List view styles */
.companies-grid.list-view .company-card {
    display: flex;
    align-items: center;
    gap: 1.5rem;
    text-align: left;
}

.companies-grid.list-view .company-logo-container {
    margin-bottom: 0;
    flex-shrink: 0;
}

.companies-grid.list-view .company-logo,
.companies-grid.list-view .company-logo-placeholder {
    width: 100px;
    height: 100px;
}

.companies-grid.list-view .company-content {
    flex: 1;
    text-align: left;
}

.companies-grid.list-view .company-footer {
    justify-content: flex-start;
}

.companies-grid.list-view .view-details-btn,
.companies-grid.list-view .contact-btn {
    flex: none;
    min-width: auto;
    padding: 0.5rem 1rem;
}

@media (max-width: 480px) {
    .company-card {
        padding: 1rem;
    }
    
    .company-footer {
        flex-direction: column;
    }
    
    .view-details-btn,
    .contact-btn {
        flex: none;
        width: 100%;
    }
}
</style>
