// Authentication ve güvenlik yardımcı fonksiyonları

export interface User {
  id: string
  email: string
  name: string
  role: 'super_admin' | 'admin' | 'editor' | 'moderator'
}

export interface Session {
  user: User
  token: string
  expiresAt: string
}

// Rol yetkileri
export const permissions = {
  super_admin: [
    'users.create',
    'users.read',
    'users.update',
    'users.delete',
    'courses.create',
    'courses.read',
    'courses.update',
    'courses.delete',
    'payments.read',
    'payments.manage',
    'settings.manage',
    'reports.view'
  ],
  admin: [
    'users.read',
    'users.update',
    'courses.create',
    'courses.read',
    'courses.update',
    'courses.delete',
    'payments.read',
    'reports.view'
  ],
  editor: [
    'courses.create',
    'courses.read',
    'courses.update',
    'reports.view'
  ],
  moderator: [
    'courses.read',
    'courses.update',
    'reviews.moderate'
  ]
}

// Kullanıcı yetkisi kontrolü
export function hasPermission(role: User['role'], permission: string): boolean {
  return permissions[role]?.includes(permission) || false
}

// Session doğrulama
export function validateSession(session: Session | null): boolean {
  if (!session) return false
  
  try {
    const expiresAt = new Date(session.expiresAt)
    return expiresAt > new Date()
  } catch {
    return false
  }
}

// Şifre güvenlik kontrolü
export function validatePassword(password: string): {
  valid: boolean
  errors: string[]
} {
  const errors: string[] = []
  
  if (password.length < 8) {
    errors.push('Şifre en az 8 karakter olmalıdır')
  }
  
  if (!/[A-Z]/.test(password)) {
    errors.push('Şifre en az bir büyük harf içermelidir')
  }
  
  if (!/[a-z]/.test(password)) {
    errors.push('Şifre en az bir küçük harf içermelidir')
  }
  
  if (!/[0-9]/.test(password)) {
    errors.push('Şifre en az bir rakam içermelidir')
  }
  
  if (!/[!@#$%^&*]/.test(password)) {
    errors.push('Şifre en az bir özel karakter içermelidir (!@#$%^&*)')
  }
  
  return {
    valid: errors.length === 0,
    errors
  }
}

// Rate limiting için basit implementasyon
const loginAttempts = new Map<string, { count: number; lastAttempt: number }>()

export function checkRateLimit(identifier: string, maxAttempts: number = 5, windowMs: number = 15 * 60 * 1000): {
  allowed: boolean
  remainingAttempts: number
} {
  const now = Date.now()
  const attempts = loginAttempts.get(identifier)
  
  if (!attempts || now - attempts.lastAttempt > windowMs) {
    // Yeni window veya zaman aşımı
    loginAttempts.set(identifier, { count: 1, lastAttempt: now })
    return { allowed: true, remainingAttempts: maxAttempts - 1 }
  }
  
  if (attempts.count >= maxAttempts) {
    // Limit aşıldı
    return { allowed: false, remainingAttempts: 0 }
  }
  
  // Sayacı artır
  attempts.count++
  attempts.lastAttempt = now
  loginAttempts.set(identifier, attempts)
  
  return { allowed: true, remainingAttempts: maxAttempts - attempts.count }
}

// XSS koruması - Input sanitization
export function sanitizeInput(input: string): string {
  return input
    .replace(/&/g, '&amp;')
    .replace(/</g, '&lt;')
    .replace(/>/g, '&gt;')
    .replace(/"/g, '&quot;')
    .replace(/'/g, '&#x27;')
    .replace(/\//g, '&#x2F;')
}

// SQL Injection koruması için parametre escape
export function escapeSQL(value: string): string {
  return value.replace(/[\0\x08\x09\x1a\n\r"'\\\%]/g, (char) => {
    switch (char) {
      case "\0": return "\\0"
      case "\x08": return "\\b"
      case "\x09": return "\\t"
      case "\x1a": return "\\z"
      case "\n": return "\\n"
      case "\r": return "\\r"
      case "\"":
      case "'":
      case "\\":
      case "%":
        return "\\" + char
      default:
        return char
    }
  })
}

// CSRF Token oluşturma
export function generateCSRFToken(): string {
  return Array.from(crypto.getRandomValues(new Uint8Array(32)))
    .map(b => b.toString(16).padStart(2, '0'))
    .join('')
}

// Token doğrulama
export function verifyCSRFToken(token: string, storedToken: string): boolean {
  if (!token || !storedToken) return false
  return token === storedToken
}

// Audit log için
export interface AuditLog {
  userId: string
  action: string
  resource: string
  timestamp: string
  ip?: string
  userAgent?: string
}

export function createAuditLog(log: AuditLog): void {
  // Gerçek uygulamada veritabanına kaydedilecek
  console.log('[AUDIT]', log)
}

