import { Metadata } from 'next'
import { generatePageTitle, generateMetaDescription, generateCanonicalUrl, generateOgImageUrl } from './seo-utils'

interface MetadataOptions {
  title: string
  description: string
  keywords?: string[]
  path: string
  image?: string
  noindex?: boolean
  type?: 'website' | 'article'
  publishedTime?: string
  modifiedTime?: string
  author?: string
}

/**
 * Generate complete metadata for a page
 */
export function generateMetadata(options: MetadataOptions): Metadata {
  const {
    title,
    description,
    keywords = [],
    path,
    image,
    noindex = false,
    type = 'website',
    publishedTime,
    modifiedTime,
    author
  } = options

  const fullTitle = generatePageTitle(title)
  const metaDescription = generateMetaDescription(description)
  const canonical = generateCanonicalUrl(path)
  const ogImage = generateOgImageUrl(image)

  const metadata: Metadata = {
    title: fullTitle,
    description: metaDescription,
    keywords: keywords.join(', '),
    authors: author ? [{ name: author }] : [{ name: 'Dans Dersleri' }],
    creator: 'Dans Dersleri',
    publisher: 'Dans Dersleri',
    alternates: {
      canonical,
    },
    openGraph: {
      title: fullTitle,
      description: metaDescription,
      url: canonical,
      siteName: 'Dans Dersleri',
      images: [
        {
          url: ogImage,
          width: 1200,
          height: 630,
          alt: title,
        },
      ],
      locale: 'tr_TR',
      type: type as any,
    },
    twitter: {
      card: 'summary_large_image',
      title: fullTitle,
      description: metaDescription,
      images: [ogImage],
      creator: '@dansdersleri',
    },
  }

  // Add article-specific metadata
  if (type === 'article' && (publishedTime || modifiedTime)) {
    metadata.openGraph = {
      ...metadata.openGraph,
      type: 'article',
      publishedTime,
      modifiedTime,
      authors: author ? [author] : undefined,
    }
  }

  // Add robots metadata
  if (noindex) {
    metadata.robots = {
      index: false,
      follow: true,
    }
  } else {
    metadata.robots = {
      index: true,
      follow: true,
      googleBot: {
        index: true,
        follow: true,
        'max-video-preview': -1,
        'max-image-preview': 'large',
        'max-snippet': -1,
      },
    }
  }

  return metadata
}

/**
 * Generate metadata for course pages
 */
export function generateCourseMetadata(course: {
  id: string
  name: string
  description: string
  city: string
  danceType: string
  price: number
  instructor: string
  image?: string
}) {
  const title = `${course.name} - ${course.city} ${course.danceType} Kursu`
  const description = `${course.city}'de ${course.danceType} öğrenmek için ${course.name}. Eğitmen: ${course.instructor}. ${course.price}₺'den başlayan fiyatlarla. ${course.description.slice(0, 100)}...`
  
  const keywords = [
    course.danceType.toLowerCase(),
    `${course.city.toLowerCase()} dans kursu`,
    `${course.danceType.toLowerCase()} dersi`,
    `${course.city.toLowerCase()} ${course.danceType.toLowerCase()}`,
    'dans dersleri',
    'dans okulu',
    course.instructor.toLowerCase(),
  ]

  return generateMetadata({
    title,
    description,
    keywords,
    path: `/kurslar/${course.id}`,
    image: course.image,
  })
}

/**
 * Generate metadata for city pages
 */
export function generateCityMetadata(city: string, courseCount: number) {
  const title = `${city} Dans Kursları - ${courseCount}+ Profesyonel Dans Dersi`
  const description = `${city}'de ${courseCount}+ profesyonel dans kursu. Salsa, Bachata, Tango, Hip-Hop ve daha fazlası. En iyi dans okulları ve öğretmenlerle tanışın.`
  
  const keywords = [
    `${city.toLowerCase()} dans kursu`,
    `${city.toLowerCase()} dans dersleri`,
    `${city.toLowerCase()} dans okulu`,
    `${city.toLowerCase()} salsa`,
    `${city.toLowerCase()} bachata`,
    `${city.toLowerCase()} tango`,
    'dans kursları',
  ]

  return generateMetadata({
    title,
    description,
    keywords,
    path: `/sehirler/${city.toLowerCase()}`,
  })
}

/**
 * Generate metadata for dance type pages
 */
export function generateDanceTypeMetadata(danceType: string, courseCount: number) {
  const title = `${danceType} Dersleri - ${courseCount}+ Kurs Seçeneği`
  const description = `Türkiye'nin en iyi ${danceType} kursları. ${courseCount}+ farklı seçenek, profesyonel öğretmenler, uygun fiyatlar. Hemen başla!`
  
  const keywords = [
    `${danceType.toLowerCase()} dersi`,
    `${danceType.toLowerCase()} kursu`,
    `${danceType.toLowerCase()} öğren`,
    `${danceType.toLowerCase()} okulu`,
    'dans dersleri',
    'dans kursu',
  ]

  return generateMetadata({
    title,
    description,
    keywords,
    path: `/dans-turleri/${danceType.toLowerCase().replace(/\s+/g, '-')}`,
  })
}

/**
 * Generate metadata for blog posts
 */
export function generateBlogMetadata(post: {
  id: string
  title: string
  excerpt: string
  author: string
  publishedAt: string
  updatedAt?: string
  image?: string
  tags: string[]
}) {
  const title = post.title
  const description = post.excerpt
  
  return generateMetadata({
    title,
    description,
    keywords: post.tags,
    path: `/blog/${post.id}`,
    image: post.image,
    type: 'article',
    publishedTime: post.publishedAt,
    modifiedTime: post.updatedAt,
    author: post.author,
  })
}

/**
 * Generate metadata for search results
 */
export function generateSearchMetadata(query: string, resultCount: number) {
  const title = `"${query}" Arama Sonuçları - ${resultCount} Kurs Bulundu`
  const description = `"${query}" için ${resultCount} dans kursu bulundu. Türkiye'nin en kapsamlı dans kursu rehberinde aradığınız kursu bulun.`
  
  const keywords = [
    query.toLowerCase(),
    'dans kursu',
    'dans dersleri',
    'dans arama',
  ]

  return generateMetadata({
    title,
    description,
    keywords,
    path: `/arama?q=${encodeURIComponent(query)}`,
    noindex: true, // Search results shouldn't be indexed
  })
}

