// Schema.org Structured Data helpers

export interface Course {
  id: string
  name: string
  description: string
  instructor: string
  price: number
  currency: string
  rating: number
  reviewCount: number
  image: string
  location: {
    address: string
    city: string
  }
}

export function generateOrganizationSchema() {
  return {
    '@context': 'https://schema.org',
    '@type': 'Organization',
    name: 'Dans Dersleri',
    description: 'Türkiye\'nin en kapsamlı dans kursu rehberi platformu',
    url: 'https://dansdersleri.com',
    logo: 'https://dansdersleri.com/logo.png',
    contactPoint: {
      '@type': 'ContactPoint',
      telephone: '+90-212-555-0123',
      contactType: 'customer service',
      areaServed: 'TR',
      availableLanguage: ['Turkish']
    },
    sameAs: [
      'https://facebook.com/dansdersleri',
      'https://instagram.com/dansdersleri',
      'https://twitter.com/dansdersleri'
    ]
  }
}

export function generateWebsiteSchema() {
  return {
    '@context': 'https://schema.org',
    '@type': 'WebSite',
    name: 'Dans Dersleri',
    url: 'https://dansdersleri.com',
    potentialAction: {
      '@type': 'SearchAction',
      target: {
        '@type': 'EntryPoint',
        urlTemplate: 'https://dansdersleri.com/arama?q={search_term_string}'
      },
      'query-input': 'required name=search_term_string'
    }
  }
}

export function generateCourseSchema(course: Course) {
  return {
    '@context': 'https://schema.org',
    '@type': 'Course',
    name: course.name,
    description: course.description,
    provider: {
      '@type': 'Organization',
      name: course.name,
      sameAs: 'https://dansdersleri.com'
    },
    instructor: {
      '@type': 'Person',
      name: course.instructor
    },
    offers: {
      '@type': 'Offer',
      price: course.price,
      priceCurrency: course.currency,
      availability: 'https://schema.org/InStock'
    },
    aggregateRating: {
      '@type': 'AggregateRating',
      ratingValue: course.rating,
      reviewCount: course.reviewCount,
      bestRating: 5,
      worstRating: 1
    },
    image: course.image,
    location: {
      '@type': 'Place',
      address: {
        '@type': 'PostalAddress',
        streetAddress: course.location.address,
        addressLocality: course.location.city,
        addressCountry: 'TR'
      }
    }
  }
}

export function generateBreadcrumbSchema(items: { name: string; url: string }[]) {
  return {
    '@context': 'https://schema.org',
    '@type': 'BreadcrumbList',
    itemListElement: items.map((item, index) => ({
      '@type': 'ListItem',
      position: index + 1,
      name: item.name,
      item: item.url
    }))
  }
}

export function generateLocalBusinessSchema(business: {
  name: string
  address: string
  city: string
  phone: string
  rating: number
  reviewCount: number
}) {
  return {
    '@context': 'https://schema.org',
    '@type': 'LocalBusiness',
    '@id': 'https://dansdersleri.com',
    name: business.name,
    telephone: business.phone,
    address: {
      '@type': 'PostalAddress',
      streetAddress: business.address,
      addressLocality: business.city,
      addressCountry: 'TR'
    },
    aggregateRating: {
      '@type': 'AggregateRating',
      ratingValue: business.rating,
      reviewCount: business.reviewCount
    },
    priceRange: '₺₺'
  }
}

export function generateFAQSchema(faqs: { question: string; answer: string }[]) {
  return {
    '@context': 'https://schema.org',
    '@type': 'FAQPage',
    mainEntity: faqs.map(faq => ({
      '@type': 'Question',
      name: faq.question,
      acceptedAnswer: {
        '@type': 'Answer',
        text: faq.answer
      }
    }))
  }
}

// Article Schema for blog posts
export function generateArticleSchema(article: {
  title: string
  description: string
  author: string
  datePublished: string
  dateModified?: string
  image: string
  url: string
}) {
  return {
    '@context': 'https://schema.org',
    '@type': 'Article',
    headline: article.title,
    description: article.description,
    image: article.image,
    author: {
      '@type': 'Person',
      name: article.author
    },
    publisher: {
      '@type': 'Organization',
      name: 'Dans Dersleri',
      logo: {
        '@type': 'ImageObject',
        url: 'https://dansdersleri.com/logo.png'
      }
    },
    datePublished: article.datePublished,
    dateModified: article.dateModified || article.datePublished,
    mainEntityOfPage: {
      '@type': 'WebPage',
      '@id': article.url
    }
  }
}

// Review Schema
export function generateReviewSchema(review: {
  itemName: string
  itemType: string
  author: string
  rating: number
  reviewBody: string
  datePublished: string
}) {
  return {
    '@context': 'https://schema.org',
    '@type': 'Review',
    itemReviewed: {
      '@type': review.itemType,
      name: review.itemName
    },
    author: {
      '@type': 'Person',
      name: review.author
    },
    reviewRating: {
      '@type': 'Rating',
      ratingValue: review.rating,
      bestRating: 5,
      worstRating: 1
    },
    reviewBody: review.reviewBody,
    datePublished: review.datePublished
  }
}

// Video Schema
export function generateVideoSchema(video: {
  name: string
  description: string
  thumbnailUrl: string
  uploadDate: string
  duration: string
  contentUrl?: string
  embedUrl?: string
}) {
  return {
    '@context': 'https://schema.org',
    '@type': 'VideoObject',
    name: video.name,
    description: video.description,
    thumbnailUrl: video.thumbnailUrl,
    uploadDate: video.uploadDate,
    duration: video.duration,
    contentUrl: video.contentUrl,
    embedUrl: video.embedUrl
  }
}

// Event Schema for dance workshops/classes
export function generateEventSchema(event: {
  name: string
  description: string
  startDate: string
  endDate: string
  location: {
    name: string
    address: string
    city: string
  }
  price: number
  currency: string
  image: string
  organizer: string
}) {
  return {
    '@context': 'https://schema.org',
    '@type': 'Event',
    name: event.name,
    description: event.description,
    startDate: event.startDate,
    endDate: event.endDate,
    eventStatus: 'https://schema.org/EventScheduled',
    eventAttendanceMode: 'https://schema.org/OfflineEventAttendanceMode',
    location: {
      '@type': 'Place',
      name: event.location.name,
      address: {
        '@type': 'PostalAddress',
        streetAddress: event.location.address,
        addressLocality: event.location.city,
        addressCountry: 'TR'
      }
    },
    image: event.image,
    offers: {
      '@type': 'Offer',
      price: event.price,
      priceCurrency: event.currency,
      availability: 'https://schema.org/InStock',
      validFrom: new Date().toISOString()
    },
    organizer: {
      '@type': 'Organization',
      name: event.organizer,
      url: 'https://dansdersleri.com'
    }
  }
}

// Aggregate Rating Schema
export function generateAggregateRatingSchema(item: {
  name: string
  ratingValue: number
  reviewCount: number
  bestRating?: number
  worstRating?: number
}) {
  return {
    '@context': 'https://schema.org',
    '@type': 'AggregateRating',
    itemReviewed: {
      '@type': 'Thing',
      name: item.name
    },
    ratingValue: item.ratingValue,
    reviewCount: item.reviewCount,
    bestRating: item.bestRating || 5,
    worstRating: item.worstRating || 1
  }
}

// HowTo Schema for dance tutorials
export function generateHowToSchema(howTo: {
  name: string
  description: string
  image: string
  totalTime: string
  steps: { name: string; text: string; image?: string }[]
}) {
  return {
    '@context': 'https://schema.org',
    '@type': 'HowTo',
    name: howTo.name,
    description: howTo.description,
    image: howTo.image,
    totalTime: howTo.totalTime,
    step: howTo.steps.map((step, index) => ({
      '@type': 'HowToStep',
      position: index + 1,
      name: step.name,
      text: step.text,
      image: step.image
    }))
  }
}

// ItemList Schema for listings
export function generateItemListSchema(items: {
  name: string
  url: string
  image?: string
}[]) {
  return {
    '@context': 'https://schema.org',
    '@type': 'ItemList',
    itemListElement: items.map((item, index) => ({
      '@type': 'ListItem',
      position: index + 1,
      item: {
        '@type': 'Thing',
        name: item.name,
        url: item.url,
        image: item.image
      }
    }))
  }
}

