<?php
/**
 * Custom Post Type and Taxonomies
 *
 * @package FirmaRehberiPro
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Register Company Post Type
 */
function firma_rehberi_register_company_post_type() {
    $labels = array(
        'name'                  => _x('Firmalar', 'Post type general name', 'firma-rehberi-pro'),
        'singular_name'         => _x('Firma', 'Post type singular name', 'firma-rehberi-pro'),
        'menu_name'             => _x('Firmalar', 'Admin Menu text', 'firma-rehberi-pro'),
        'name_admin_bar'        => _x('Firma', 'Add New on Toolbar', 'firma-rehberi-pro'),
        'add_new'               => __('Yeni Ekle', 'firma-rehberi-pro'),
        'add_new_item'          => __('Yeni Firma Ekle', 'firma-rehberi-pro'),
        'new_item'              => __('Yeni Firma', 'firma-rehberi-pro'),
        'edit_item'             => __('Firmayı Düzenle', 'firma-rehberi-pro'),
        'view_item'             => __('Firmayı Görüntüle', 'firma-rehberi-pro'),
        'all_items'             => __('Tüm Firmalar', 'firma-rehberi-pro'),
        'search_items'          => __('Firma Ara', 'firma-rehberi-pro'),
        'parent_item_colon'     => __('Ana Firma:', 'firma-rehberi-pro'),
        'not_found'             => __('Firma bulunamadı.', 'firma-rehberi-pro'),
        'not_found_in_trash'    => __('Çöp kutusunda firma bulunamadı.', 'firma-rehberi-pro'),
        'featured_image'        => _x('Firma Logosu', 'Overrides the "Featured Image" phrase', 'firma-rehberi-pro'),
        'set_featured_image'    => _x('Firma logosu ayarla', 'Overrides the "Set featured image" phrase', 'firma-rehberi-pro'),
        'remove_featured_image' => _x('Firma logosunu kaldır', 'Overrides the "Remove featured image" phrase', 'firma-rehberi-pro'),
        'use_featured_image'    => _x('Firma logosu olarak kullan', 'Overrides the "Use as featured image" phrase', 'firma-rehberi-pro'),
        'archives'              => _x('Firma arşivleri', 'The post type archive label', 'firma-rehberi-pro'),
        'insert_into_item'      => _x('Firmaya ekle', 'Overrides the "Insert into post" phrase', 'firma-rehberi-pro'),
        'uploaded_to_this_item' => _x('Bu firmaya yüklendi', 'Overrides the "Uploaded to this post" phrase', 'firma-rehberi-pro'),
        'filter_items_list'     => _x('Firmaları filtrele', 'Screen reader text for the filter links', 'firma-rehberi-pro'),
        'items_list_navigation' => _x('Firma listesi navigasyonu', 'Screen reader text for the pagination', 'firma-rehberi-pro'),
        'items_list'            => _x('Firma listesi', 'Screen reader text for the items list', 'firma-rehberi-pro'),
    );

    $args = array(
        'labels'             => $labels,
        'public'             => true,
        'publicly_queryable' => true,
        'show_ui'            => true,
        'show_in_menu'       => true,
        'show_in_nav_menus'  => true,
        'show_in_admin_bar'  => true,
        'show_in_rest'       => true,
        'query_var'          => true,
        'rewrite'            => array('slug' => 'firma'),
        'capability_type'    => 'post',
        'has_archive'        => true,
        'hierarchical'       => false,
        'menu_position'      => 5,
        'menu_icon'          => 'dashicons-building',
        'supports'           => array('title', 'editor', 'excerpt', 'thumbnail', 'comments', 'custom-fields'),
        'taxonomies'         => array('company_category', 'company_city'),
    );

    register_post_type('company', $args);
}
add_action('init', 'firma_rehberi_register_company_post_type');

/**
 * Register Company Category Taxonomy
 */
function firma_rehberi_register_company_category_taxonomy() {
    $labels = array(
        'name'                       => _x('Firma Kategorileri', 'Taxonomy General Name', 'firma-rehberi-pro'),
        'singular_name'              => _x('Firma Kategorisi', 'Taxonomy Singular Name', 'firma-rehberi-pro'),
        'menu_name'                  => __('Kategoriler', 'firma-rehberi-pro'),
        'all_items'                  => __('Tüm Kategoriler', 'firma-rehberi-pro'),
        'parent_item'                => __('Ana Kategori', 'firma-rehberi-pro'),
        'parent_item_colon'          => __('Ana Kategori:', 'firma-rehberi-pro'),
        'new_item_name'              => __('Yeni Kategori Adı', 'firma-rehberi-pro'),
        'add_new_item'               => __('Yeni Kategori Ekle', 'firma-rehberi-pro'),
        'edit_item'                  => __('Kategoriyi Düzenle', 'firma-rehberi-pro'),
        'update_item'                => __('Kategoriyi Güncelle', 'firma-rehberi-pro'),
        'view_item'                  => __('Kategoriyi Görüntüle', 'firma-rehberi-pro'),
        'separate_items_with_commas' => __('Kategorileri virgülle ayırın', 'firma-rehberi-pro'),
        'add_or_remove_items'        => __('Kategori ekle veya kaldır', 'firma-rehberi-pro'),
        'choose_from_most_used'      => __('En çok kullanılanlardan seç', 'firma-rehberi-pro'),
        'popular_items'              => __('Popüler Kategoriler', 'firma-rehberi-pro'),
        'search_items'               => __('Kategori Ara', 'firma-rehberi-pro'),
        'not_found'                  => __('Kategori bulunamadı', 'firma-rehberi-pro'),
        'no_terms'                   => __('Kategori yok', 'firma-rehberi-pro'),
        'items_list'                 => __('Kategori listesi', 'firma-rehberi-pro'),
        'items_list_navigation'      => __('Kategori listesi navigasyonu', 'firma-rehberi-pro'),
    );

    $args = array(
        'labels'                     => $labels,
        'hierarchical'               => true,
        'public'                     => true,
        'show_ui'                    => true,
        'show_admin_column'          => true,
        'show_in_nav_menus'          => true,
        'show_in_rest'               => true,
        'show_tagcloud'              => true,
        'rewrite'                    => array('slug' => 'kategori'),
    );

    register_taxonomy('company_category', array('company'), $args);
}
add_action('init', 'firma_rehberi_register_company_category_taxonomy');

/**
 * Register Company City Taxonomy
 */
function firma_rehberi_register_company_city_taxonomy() {
    $labels = array(
        'name'                       => _x('Şehirler', 'Taxonomy General Name', 'firma-rehberi-pro'),
        'singular_name'              => _x('Şehir', 'Taxonomy Singular Name', 'firma-rehberi-pro'),
        'menu_name'                  => __('Şehirler', 'firma-rehberi-pro'),
        'all_items'                  => __('Tüm Şehirler', 'firma-rehberi-pro'),
        'parent_item'                => __('Ana Şehir', 'firma-rehberi-pro'),
        'parent_item_colon'          => __('Ana Şehir:', 'firma-rehberi-pro'),
        'new_item_name'              => __('Yeni Şehir Adı', 'firma-rehberi-pro'),
        'add_new_item'               => __('Yeni Şehir Ekle', 'firma-rehberi-pro'),
        'edit_item'                  => __('Şehri Düzenle', 'firma-rehberi-pro'),
        'update_item'                => __('Şehri Güncelle', 'firma-rehberi-pro'),
        'view_item'                  => __('Şehri Görüntüle', 'firma-rehberi-pro'),
        'separate_items_with_commas' => __('Şehirleri virgülle ayırın', 'firma-rehberi-pro'),
        'add_or_remove_items'        => __('Şehir ekle veya kaldır', 'firma-rehberi-pro'),
        'choose_from_most_used'      => __('En çok kullanılanlardan seç', 'firma-rehberi-pro'),
        'popular_items'              => __('Popüler Şehirler', 'firma-rehberi-pro'),
        'search_items'               => __('Şehir Ara', 'firma-rehberi-pro'),
        'not_found'                  => __('Şehir bulunamadı', 'firma-rehberi-pro'),
        'no_terms'                   => __('Şehir yok', 'firma-rehberi-pro'),
        'items_list'                 => __('Şehir listesi', 'firma-rehberi-pro'),
        'items_list_navigation'      => __('Şehir listesi navigasyonu', 'firma-rehberi-pro'),
    );

    $args = array(
        'labels'                     => $labels,
        'hierarchical'               => false,
        'public'                     => true,
        'show_ui'                    => true,
        'show_admin_column'          => true,
        'show_in_nav_menus'          => true,
        'show_in_rest'               => true,
        'show_tagcloud'              => true,
        'rewrite'                    => array('slug' => 'sehir'),
    );

    register_taxonomy('company_city', array('company'), $args);
}
add_action('init', 'firma_rehberi_register_company_city_taxonomy');

/**
 * Add custom meta boxes for company post type
 */
function firma_rehberi_add_company_meta_boxes() {
    add_meta_box(
        'company_contact_info',
        __('İletişim Bilgileri', 'firma-rehberi-pro'),
        'firma_rehberi_company_contact_info_callback',
        'company',
        'normal',
        'high'
    );
    
    add_meta_box(
        'company_business_info',
        __('İşletme Bilgileri', 'firma-rehberi-pro'),
        'firma_rehberi_company_business_info_callback',
        'company',
        'normal',
        'high'
    );
    
    add_meta_box(
        'company_social_media',
        __('Sosyal Medya', 'firma-rehberi-pro'),
        'firma_rehberi_company_social_media_callback',
        'company',
        'normal',
        'default'
    );
    
    add_meta_box(
        'company_gallery',
        __('Galeri', 'firma-rehberi-pro'),
        'firma_rehberi_company_gallery_callback',
        'company',
        'normal',
        'default'
    );
}
add_action('add_meta_boxes', 'firma_rehberi_add_company_meta_boxes');

/**
 * Contact info meta box callback
 */
function firma_rehberi_company_contact_info_callback($post) {
    wp_nonce_field('firma_rehberi_company_meta', 'firma_rehberi_company_meta_nonce');
    
    $phone = get_post_meta($post->ID, '_phone', true);
    $email = get_post_meta($post->ID, '_email', true);
    $website = get_post_meta($post->ID, '_website', true);
    $address = get_post_meta($post->ID, '_address', true);
    $working_hours = get_post_meta($post->ID, '_working_hours', true);
    ?>
    <table class="form-table">
        <tr>
            <th scope="row">
                <label for="company_phone"><?php _e('Telefon', 'firma-rehberi-pro'); ?></label>
            </th>
            <td>
                <input type="tel" id="company_phone" name="company_phone" value="<?php echo esc_attr($phone); ?>" class="regular-text" />
            </td>
        </tr>
        <tr>
            <th scope="row">
                <label for="company_email"><?php _e('E-posta', 'firma-rehberi-pro'); ?></label>
            </th>
            <td>
                <input type="email" id="company_email" name="company_email" value="<?php echo esc_attr($email); ?>" class="regular-text" />
            </td>
        </tr>
        <tr>
            <th scope="row">
                <label for="company_website"><?php _e('Website', 'firma-rehberi-pro'); ?></label>
            </th>
            <td>
                <input type="url" id="company_website" name="company_website" value="<?php echo esc_attr($website); ?>" class="regular-text" placeholder="https://" />
            </td>
        </tr>
        <tr>
            <th scope="row">
                <label for="company_address"><?php _e('Adres', 'firma-rehberi-pro'); ?></label>
            </th>
            <td>
                <textarea id="company_address" name="company_address" rows="3" class="large-text"><?php echo esc_textarea($address); ?></textarea>
            </td>
        </tr>
        <tr>
            <th scope="row">
                <label for="company_working_hours"><?php _e('Çalışma Saatleri', 'firma-rehberi-pro'); ?></label>
            </th>
            <td>
                <input type="text" id="company_working_hours" name="company_working_hours" value="<?php echo esc_attr($working_hours); ?>" class="regular-text" placeholder="Pazartesi-Cuma: 09:00-18:00" />
            </td>
        </tr>
    </table>
    <?php
}

/**
 * Business info meta box callback
 */
function firma_rehberi_company_business_info_callback($post) {
    $rating = get_post_meta($post->ID, '_rating', true);
    $featured = get_post_meta($post->ID, '_featured', true);
    $services = get_post_meta($post->ID, '_services', true);
    $latitude = get_post_meta($post->ID, '_latitude', true);
    $longitude = get_post_meta($post->ID, '_longitude', true);
    ?>
    <table class="form-table">
        <tr>
            <th scope="row">
                <label for="company_rating"><?php _e('Değerlendirme (1-5)', 'firma-rehberi-pro'); ?></label>
            </th>
            <td>
                <input type="number" id="company_rating" name="company_rating" value="<?php echo esc_attr($rating); ?>" min="1" max="5" step="0.1" class="small-text" />
            </td>
        </tr>
        <tr>
            <th scope="row">
                <label for="company_featured"><?php _e('Öne Çıkan Firma', 'firma-rehberi-pro'); ?></label>
            </th>
            <td>
                <input type="checkbox" id="company_featured" name="company_featured" value="1" <?php checked($featured, '1'); ?> />
                <label for="company_featured"><?php _e('Ana sayfada öne çıkar', 'firma-rehberi-pro'); ?></label>
            </td>
        </tr>
        <tr>
            <th scope="row">
                <label for="company_services"><?php _e('Hizmetler', 'firma-rehberi-pro'); ?></label>
            </th>
            <td>
                <textarea id="company_services" name="company_services" rows="5" class="large-text" placeholder="<?php _e('Her satıra bir hizmet yazın', 'firma-rehberi-pro'); ?>"><?php echo esc_textarea($services); ?></textarea>
            </td>
        </tr>
        <tr>
            <th scope="row">
                <label for="company_latitude"><?php _e('Enlem', 'firma-rehberi-pro'); ?></label>
            </th>
            <td>
                <input type="number" id="company_latitude" name="company_latitude" value="<?php echo esc_attr($latitude); ?>" step="any" class="regular-text" placeholder="41.0082" />
            </td>
        </tr>
        <tr>
            <th scope="row">
                <label for="company_longitude"><?php _e('Boylam', 'firma-rehberi-pro'); ?></label>
            </th>
            <td>
                <input type="number" id="company_longitude" name="company_longitude" value="<?php echo esc_attr($longitude); ?>" step="any" class="regular-text" placeholder="28.9784" />
            </td>
        </tr>
    </table>
    <?php
}

/**
 * Social media meta box callback
 */
function firma_rehberi_company_social_media_callback($post) {
    $facebook = get_post_meta($post->ID, '_facebook', true);
    $twitter = get_post_meta($post->ID, '_twitter', true);
    $instagram = get_post_meta($post->ID, '_instagram', true);
    $linkedin = get_post_meta($post->ID, '_linkedin', true);
    ?>
    <table class="form-table">
        <tr>
            <th scope="row">
                <label for="company_facebook"><?php _e('Facebook', 'firma-rehberi-pro'); ?></label>
            </th>
            <td>
                <input type="url" id="company_facebook" name="company_facebook" value="<?php echo esc_attr($facebook); ?>" class="regular-text" placeholder="https://facebook.com/username" />
            </td>
        </tr>
        <tr>
            <th scope="row">
                <label for="company_twitter"><?php _e('Twitter', 'firma-rehberi-pro'); ?></label>
            </th>
            <td>
                <input type="url" id="company_twitter" name="company_twitter" value="<?php echo esc_attr($twitter); ?>" class="regular-text" placeholder="https://twitter.com/username" />
            </td>
        </tr>
        <tr>
            <th scope="row">
                <label for="company_instagram"><?php _e('Instagram', 'firma-rehberi-pro'); ?></label>
            </th>
            <td>
                <input type="url" id="company_instagram" name="company_instagram" value="<?php echo esc_attr($instagram); ?>" class="regular-text" placeholder="https://instagram.com/username" />
            </td>
        </tr>
        <tr>
            <th scope="row">
                <label for="company_linkedin"><?php _e('LinkedIn', 'firma-rehberi-pro'); ?></label>
            </th>
            <td>
                <input type="url" id="company_linkedin" name="company_linkedin" value="<?php echo esc_attr($linkedin); ?>" class="regular-text" placeholder="https://linkedin.com/company/username" />
            </td>
        </tr>
    </table>
    <?php
}

/**
 * Gallery meta box callback
 */
function firma_rehberi_company_gallery_callback($post) {
    $gallery = get_post_meta($post->ID, '_gallery', true);
    ?>
    <div class="gallery-metabox">
        <input type="hidden" id="company_gallery" name="company_gallery" value="<?php echo esc_attr($gallery); ?>" />
        <div class="gallery-preview" id="gallery-preview">
            <?php
            if ($gallery) {
                $gallery_ids = explode(',', $gallery);
                foreach ($gallery_ids as $image_id) {
                    if ($image_id) {
                        echo wp_get_attachment_image($image_id, 'thumbnail', false, array('class' => 'gallery-thumbnail'));
                    }
                }
            }
            ?>
        </div>
        <button type="button" class="button" id="add-gallery-images"><?php _e('Galeri Ekle', 'firma-rehberi-pro'); ?></button>
        <button type="button" class="button" id="clear-gallery" style="display: <?php echo $gallery ? 'inline-block' : 'none'; ?>"><?php _e('Galeriyi Temizle', 'firma-rehberi-pro'); ?></button>
    </div>
    
    <script>
    jQuery(document).ready(function($) {
        var galleryFrame;
        
        $('#add-gallery-images').on('click', function(e) {
            e.preventDefault();
            
            if (galleryFrame) {
                galleryFrame.open();
                return;
            }
            
            galleryFrame = wp.media({
                title: '<?php _e('Galeri Seç', 'firma-rehberi-pro'); ?>',
                button: {
                    text: '<?php _e('Seç', 'firma-rehberi-pro'); ?>'
                },
                multiple: true
            });
            
            galleryFrame.on('select', function() {
                var selection = galleryFrame.state().get('selection');
                var ids = [];
                var preview = $('#gallery-preview');
                
                selection.map(function(attachment) {
                    ids.push(attachment.id);
                    preview.append(attachment.attributes.sizes.thumbnail ? 
                        $('<img>').attr('src', attachment.attributes.sizes.thumbnail.url).addClass('gallery-thumbnail') :
                        $('<img>').attr('src', attachment.attributes.url).addClass('gallery-thumbnail')
                    );
                });
                
                $('#company_gallery').val(ids.join(','));
                $('#clear-gallery').show();
            });
            
            galleryFrame.open();
        });
        
        $('#clear-gallery').on('click', function(e) {
            e.preventDefault();
            $('#company_gallery').val('');
            $('#gallery-preview').empty();
            $(this).hide();
        });
    });
    </script>
    
    <style>
    .gallery-metabox .gallery-preview {
        display: flex;
        flex-wrap: wrap;
        gap: 10px;
        margin-bottom: 10px;
    }
    
    .gallery-metabox .gallery-thumbnail {
        width: 80px;
        height: 80px;
        object-fit: cover;
        border-radius: 4px;
        border: 1px solid #ddd;
    }
    </style>
    <?php
}

/**
 * Save company meta data
 */
function firma_rehberi_save_company_meta($post_id) {
    if (!isset($_POST['firma_rehberi_company_meta_nonce']) || !wp_verify_nonce($_POST['firma_rehberi_company_meta_nonce'], 'firma_rehberi_company_meta')) {
        return;
    }
    
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
        return;
    }
    
    if (!current_user_can('edit_post', $post_id)) {
        return;
    }
    
    // Contact info
    $fields = array(
        '_phone' => 'company_phone',
        '_email' => 'company_email',
        '_website' => 'company_website',
        '_address' => 'company_address',
        '_working_hours' => 'company_working_hours',
        '_rating' => 'company_rating',
        '_featured' => 'company_featured',
        '_services' => 'company_services',
        '_latitude' => 'company_latitude',
        '_longitude' => 'company_longitude',
        '_facebook' => 'company_facebook',
        '_twitter' => 'company_twitter',
        '_instagram' => 'company_instagram',
        '_linkedin' => 'company_linkedin',
        '_gallery' => 'company_gallery',
    );
    
    foreach ($fields as $meta_key => $field_name) {
        if (isset($_POST[$field_name])) {
            $value = $_POST[$field_name];
            
            // Sanitize based on field type
            if (in_array($field_name, array('company_phone', 'company_working_hours'))) {
                $value = sanitize_text_field($value);
            } elseif (in_array($field_name, array('company_email'))) {
                $value = sanitize_email($value);
            } elseif (in_array($field_name, array('company_website', 'company_facebook', 'company_twitter', 'company_instagram', 'company_linkedin'))) {
                $value = esc_url_raw($value);
            } elseif (in_array($field_name, array('company_address', 'company_services'))) {
                $value = sanitize_textarea_field($value);
            } elseif (in_array($field_name, array('company_rating', 'company_latitude', 'company_longitude'))) {
                $value = floatval($value);
            } elseif ($field_name === 'company_featured') {
                $value = '1';
            } else {
                $value = sanitize_text_field($value);
            }
            
            update_post_meta($post_id, $meta_key, $value);
        } else {
            // Handle unchecked checkboxes
            if ($field_name === 'company_featured') {
                delete_post_meta($post_id, $meta_key);
            }
        }
    }
}
add_action('save_post', 'firma_rehberi_save_company_meta');

/**
 * Register meta fields for REST API
 */
function firma_rehberi_register_company_meta_fields() {
    $meta_fields = array(
        '_phone',
        '_email',
        '_website',
        '_address',
        '_working_hours',
        '_rating',
        '_featured',
        '_services',
        '_latitude',
        '_longitude',
        '_facebook',
        '_twitter',
        '_instagram',
        '_linkedin',
        '_gallery',
        '_view_count',
    );
    
    foreach ($meta_fields as $field) {
        register_meta('post', $field, array(
            'type' => 'string',
            'single' => true,
            'show_in_rest' => true,
        ));
    }
}
add_action('init', 'firma_rehberi_register_company_meta_fields');
