<?php
/**
 * Enqueue Scripts and Styles
 *
 * @package FirmaRehberiPro
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Enqueue styles and scripts
 */
function firma_rehberi_enqueue_assets() {
    // Get theme version
    $theme_version = wp_get_theme()->get('Version');
    
    // Enqueue main stylesheet
    wp_enqueue_style(
        'firma-rehberi-style',
        get_stylesheet_uri(),
        array(),
        $theme_version
    );
    
    // Enqueue Google Fonts
    wp_enqueue_style(
        'firma-rehberi-fonts',
        'https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap',
        array(),
        null
    );
    
    // Enqueue main JavaScript
    wp_enqueue_script(
        'firma-rehberi-script',
        get_template_directory_uri() . '/assets/js/main.js',
        array(),
        $theme_version,
        true
    );
    
    // Localize script for AJAX
    wp_localize_script('firma-rehberi-script', 'firmaRehberi', array(
        'ajaxUrl' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('firma_rehberi_nonce'),
        'strings' => array(
            'loading' => __('Yükleniyor...', 'firma-rehberi-pro'),
            'error' => __('Bir hata oluştu. Lütfen tekrar deneyin.', 'firma-rehberi-pro'),
            'noResults' => __('Sonuç bulunamadı.', 'firma-rehberi-pro'),
        )
    ));
    
    // Enqueue comment reply script
    if (is_singular() && comments_open() && get_option('thread_comments')) {
        wp_enqueue_script('comment-reply');
    }
    
    // Enqueue search script only on search pages
    if (is_search() || is_post_type_archive('company')) {
        wp_enqueue_script(
            'firma-rehberi-search',
            get_template_directory_uri() . '/assets/js/search.js',
            array('jquery'),
            $theme_version,
            true
        );
    }
    
    // Enqueue map script only on single company pages
    if (is_singular('company')) {
        $google_maps_api_key = get_theme_mod('google_maps_api_key');
        if ($google_maps_api_key) {
            wp_enqueue_script(
                'google-maps',
                'https://maps.googleapis.com/maps/api/js?key=' . $google_maps_api_key . '&libraries=places',
                array(),
                null,
                true
            );
        }
    }
}
add_action('wp_enqueue_scripts', 'firma_rehberi_enqueue_assets');

/**
 * Enqueue admin styles and scripts
 */
function firma_rehberi_enqueue_admin_assets($hook) {
    global $post_type;
    
    // Only load on company edit pages
    if ($post_type === 'company' && ($hook === 'post.php' || $hook === 'post-new.php')) {
        wp_enqueue_style(
            'firma-rehberi-admin',
            get_template_directory_uri() . '/assets/css/admin.css',
            array(),
            FIRMA_REHBERI_VERSION
        );
        
        wp_enqueue_script(
            'firma-rehberi-admin',
            get_template_directory_uri() . '/assets/js/admin.js',
            array('jquery', 'media-upload', 'thickbox'),
            FIRMA_REHBERI_VERSION,
            true
        );
    }
}
add_action('admin_enqueue_scripts', 'firma_rehberi_enqueue_admin_assets');

/**
 * Add critical CSS inline
 */
function firma_rehberi_add_critical_css() {
    $critical_css = '
        /* Critical CSS for above-the-fold content */
        body{font-family:-apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,Oxygen,Ubuntu,Cantarell,sans-serif;line-height:1.6;color:#1a1a1a;background-color:#fff}
        .container{max-width:1200px;margin:0 auto;padding:0 1rem}
        .site-header{background:#fff;box-shadow:0 2px 4px rgba(0,0,0,0.1);position:sticky;top:0;z-index:1000}
        .hero-section{background:linear-gradient(135deg,#1e40af 0%,#3b82f6 100%);color:#fff;padding:4rem 0;text-align:center}
        .hero-title{font-size:3rem;font-weight:800;margin-bottom:1rem;line-height:1.1}
        .search-form{max-width:600px;margin:0 auto;display:flex;gap:0.5rem;background:#fff;padding:0.5rem;border-radius:50px;box-shadow:0 10px 25px rgba(0,0,0,0.1)}
        .search-input{flex:1;border:none;padding:1rem 1.5rem;font-size:1rem;border-radius:25px;outline:none}
        .search-button{background:#1e40af;color:#fff;border:none;padding:1rem 2rem;border-radius:25px;font-weight:600;cursor:pointer}
    ';
    
    echo '<style id="firma-rehberi-critical-css">' . $critical_css . '</style>';
}
add_action('wp_head', 'firma_rehberi_add_critical_css', 1);

/**
 * Preload important resources
 */
function firma_rehberi_preload_resources() {
    // Preload Google Fonts
    echo '<link rel="preload" href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700;800&display=swap" as="style" onload="this.onload=null;this.rel=\'stylesheet\'">' . "\n";
    
    // Preload critical images
    if (is_singular('company') && has_post_thumbnail()) {
        $image_url = get_the_post_thumbnail_url(get_the_ID(), 'large');
        echo '<link rel="preload" as="image" href="' . esc_url($image_url) . '">' . "\n";
    }
    
    // Preload main JavaScript
    echo '<link rel="preload" href="' . get_template_directory_uri() . '/assets/js/main.js" as="script">' . "\n";
}
add_action('wp_head', 'firma_rehberi_preload_resources', 2);

/**
 * Add resource hints
 */
function firma_rehberi_add_resource_hints($urls, $relation_type) {
    if ($relation_type === 'preconnect') {
        $urls[] = 'https://fonts.googleapis.com';
        $urls[] = 'https://fonts.gstatic.com';
    }
    
    if ($relation_type === 'dns-prefetch') {
        $urls[] = '//fonts.googleapis.com';
        $urls[] = '//fonts.gstatic.com';
    }
    
    return $urls;
}
add_filter('wp_resource_hints', 'firma_rehberi_add_resource_hints', 10, 2);

/**
 * Optimize script loading
 */
function firma_rehberi_optimize_script_loading($tag, $handle, $src) {
    // Defer non-critical scripts
    $defer_scripts = array(
        'firma-rehberi-script',
        'firma-rehberi-search',
    );
    
    if (in_array($handle, $defer_scripts)) {
        return str_replace('<script ', '<script defer ', $tag);
    }
    
    // Async external scripts
    $async_scripts = array(
        'google-maps',
    );
    
    if (in_array($handle, $async_scripts)) {
        return str_replace('<script ', '<script async ', $tag);
    }
    
    return $tag;
}
add_filter('script_loader_tag', 'firma_rehberi_optimize_script_loading', 10, 3);

/**
 * Add performance optimizations
 */
function firma_rehberi_performance_optimizations() {
    // Remove unnecessary WordPress features
    remove_action('wp_head', 'wp_generator');
    remove_action('wp_head', 'wlwmanifest_link');
    remove_action('wp_head', 'rsd_link');
    remove_action('wp_head', 'wp_shortlink_wp_head');
    remove_action('wp_head', 'adjacent_posts_rel_link_wp_head');
    
    // Remove emoji scripts
    remove_action('wp_head', 'print_emoji_detection_script', 7);
    remove_action('wp_print_styles', 'print_emoji_styles');
    remove_action('admin_print_scripts', 'print_emoji_detection_script');
    remove_action('admin_print_styles', 'print_emoji_styles');
    
    // Remove block library CSS if not using Gutenberg
    if (!is_admin()) {
        wp_dequeue_style('wp-block-library');
        wp_dequeue_style('wp-block-library-theme');
        wp_dequeue_style('wc-block-style');
    }
}
add_action('init', 'firma_rehberi_performance_optimizations');

/**
 * Add cache headers
 */
function firma_rehberi_add_cache_headers() {
    if (!is_admin()) {
        // Cache static assets for 1 year
        if (is_singular('company') || is_post_type_archive('company')) {
            header('Cache-Control: public, max-age=31536000');
        }
    }
}
add_action('send_headers', 'firma_rehberi_add_cache_headers');

/**
 * Minify inline CSS
 */
function firma_rehberi_minify_css($css) {
    // Remove comments
    $css = preg_replace('!/\*[^*]*\*+([^/][^*]*\*+)*/!', '', $css);
    
    // Remove unnecessary whitespace
    $css = str_replace(array("\r\n", "\r", "\n", "\t", '  ', '    ', '    '), '', $css);
    
    // Remove spaces around specific characters
    $css = str_replace(array(' {', '{ ', ' }', '} ', '; ', ' ;', ': ', ' :', ', ', ' ,'), array('{', '{', '}', '}', ';', ';', ':', ':', ',', ','), $css);
    
    return $css;
}

/**
 * Add service worker for caching
 */
function firma_rehberi_add_service_worker() {
    if (is_front_page()) {
        echo '<script>
        if ("serviceWorker" in navigator) {
            window.addEventListener("load", function() {
                navigator.serviceWorker.register("' . get_template_directory_uri() . '/sw.js")
                .then(function(registration) {
                    console.log("ServiceWorker registration successful");
                })
                .catch(function(err) {
                    console.log("ServiceWorker registration failed");
                });
            });
        }
        </script>';
    }
}
add_action('wp_footer', 'firma_rehberi_add_service_worker');
