<?php
/**
 * PHP 8.1 Performance Optimizations
 *
 * @package FirmaRehberiPro
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * PHP 8.1 specific performance optimizations
 */
function firma_rehberi_php81_performance_init() {
    // Enable OPcache optimizations
    if (function_exists('opcache_get_status')) {
        $opcache_status = opcache_get_status();
        if ($opcache_status && $opcache_status['opcache_enabled']) {
            // OPcache is enabled, optimize for it
            add_action('init', 'firma_rehberi_optimize_opcache');
        }
    }
    
    // Memory optimizations
    add_action('init', 'firma_rehberi_memory_optimizations');
    
    // Database query optimizations
    add_action('init', 'firma_rehberi_query_optimizations');
    
    // Asset loading optimizations
    add_action('wp_enqueue_scripts', 'firma_rehberi_asset_optimizations', 1);
}
add_action('after_setup_theme', 'firma_rehberi_php81_performance_init');

/**
 * OPcache optimizations for PHP 8.1
 */
function firma_rehberi_optimize_opcache() {
    // Set optimal OPcache settings for theme
    if (function_exists('ini_set')) {
        // Increase OPcache memory
        ini_set('opcache.memory_consumption', '128');
        
        // Enable OPcache file cache
        ini_set('opcache.file_cache', '/tmp/opcache');
        
        // Optimize for production
        ini_set('opcache.optimization_level', '0x7FFFBFFF');
        
        // Enable OPcache validate timestamps
        ini_set('opcache.validate_timestamps', '0');
    }
}

/**
 * Memory optimizations for PHP 8.1
 */
function firma_rehberi_memory_optimizations() {
    // Set optimal memory limit
    $current_limit = ini_get('memory_limit');
    $limit_bytes = wp_convert_hr_to_bytes($current_limit);
    
    // Increase memory limit if needed
    if ($limit_bytes < wp_convert_hr_to_bytes('256M')) {
        ini_set('memory_limit', '256M');
    }
    
    // Enable garbage collection
    if (function_exists('gc_enable')) {
        gc_enable();
    }
    
    // Set garbage collection threshold
    if (function_exists('gc_threshold')) {
        gc_threshold(10000);
    }
}

/**
 * Database query optimizations for PHP 8.1
 */
function firma_rehberi_query_optimizations() {
    // Optimize WordPress queries
    add_action('pre_get_posts', 'firma_rehberi_optimize_queries');
    
    // Cache expensive queries
    add_action('init', 'firma_rehberi_cache_queries');
    
    // Optimize meta queries
    add_filter('posts_where', 'firma_rehberi_optimize_meta_queries', 10, 2);
}

/**
 * Optimize WordPress queries
 */
function firma_rehberi_optimize_queries($query) {
    if (!is_admin() && $query->is_main_query()) {
        // Optimize company queries
        if (is_post_type_archive('company')) {
            $query->set('posts_per_page', 12);
            $query->set('no_found_rows', false);
            $query->set('update_post_meta_cache', true);
            $query->set('update_post_term_cache', true);
        }
        
        // Optimize search queries
        if (is_search()) {
            $query->set('posts_per_page', 10);
            $query->set('no_found_rows', false);
        }
    }
}

/**
 * Cache expensive queries
 */
function firma_rehberi_cache_queries() {
    // Cache company counts
    add_action('wp_ajax_get_company_count', 'firma_rehberi_get_cached_company_count');
    add_action('wp_ajax_nopriv_get_company_count', 'firma_rehberi_get_cached_company_count');
    
    // Cache category counts
    add_action('wp_ajax_get_category_count', 'firma_rehberi_get_cached_category_count');
    add_action('wp_ajax_nopriv_get_category_count', 'firma_rehberi_get_cached_category_count');
}

/**
 * Get cached company count
 */
function firma_rehberi_get_cached_company_count() {
    $cache_key = 'company_count_' . get_current_blog_id();
    $count = wp_cache_get($cache_key, 'firma_rehberi');
    
    if ($count === false) {
        $count = wp_count_posts('company');
        wp_cache_set($cache_key, $count, 'firma_rehberi', HOUR_IN_SECONDS);
    }
    
    wp_send_json_success($count);
}

/**
 * Get cached category count
 */
function firma_rehberi_get_cached_category_count() {
    $cache_key = 'category_count_' . get_current_blog_id();
    $count = wp_cache_get($cache_key, 'firma_rehberi');
    
    if ($count === false) {
        $terms = get_terms(array(
            'taxonomy' => 'company_category',
            'hide_empty' => true,
            'fields' => 'count'
        ));
        $count = is_wp_error($terms) ? 0 : $terms;
        wp_cache_set($cache_key, $count, 'firma_rehberi', HOUR_IN_SECONDS);
    }
    
    wp_send_json_success($count);
}

/**
 * Optimize meta queries
 */
function firma_rehberi_optimize_meta_queries($where, $query) {
    global $wpdb;
    
    // Optimize company meta queries
    if (is_post_type_archive('company') || is_tax('company_category') || is_tax('company_city')) {
        // Add index hints for better performance
        $where = str_replace(
            "{$wpdb->posts}.post_type = 'company'",
            "{$wpdb->posts}.post_type = 'company' AND {$wpdb->posts}.post_status = 'publish'",
            $where
        );
    }
    
    return $where;
}

/**
 * Asset loading optimizations for PHP 8.1
 */
function firma_rehberi_asset_optimizations() {
    // Preload critical resources
    add_action('wp_head', 'firma_rehberi_preload_critical_resources', 1);
    
    // Defer non-critical JavaScript
    add_filter('script_loader_tag', 'firma_rehberi_defer_scripts', 10, 3);
    
    // Optimize CSS loading
    add_filter('style_loader_tag', 'firma_rehberi_optimize_css_loading', 10, 4);
}

/**
 * Preload critical resources
 */
function firma_rehberi_preload_critical_resources() {
    // Preload critical CSS
    echo '<link rel="preload" href="' . get_template_directory_uri() . '/style.css" as="style" onload="this.onload=null;this.rel=\'stylesheet\'">';
    
    // Preload critical fonts
    echo '<link rel="preload" href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" as="style" onload="this.onload=null;this.rel=\'stylesheet\'">';
    
    // Preload critical JavaScript
    echo '<link rel="preload" href="' . get_template_directory_uri() . '/assets/js/main.js" as="script">';
}

/**
 * Defer non-critical JavaScript
 */
function firma_rehberi_defer_scripts($tag, $handle, $src) {
    // Defer non-critical scripts
    $defer_scripts = array('main', 'lazy-load', 'search');
    
    if (in_array($handle, $defer_scripts)) {
        return str_replace('<script ', '<script defer ', $tag);
    }
    
    return $tag;
}

/**
 * Optimize CSS loading
 */
function firma_rehberi_optimize_css_loading($html, $handle, $href, $media) {
    // Add media attribute for non-critical CSS
    if ($handle === 'firma-rehberi-style') {
        return str_replace("rel='stylesheet'", "rel='stylesheet' media='print' onload='this.media=\"all\"'", $html);
    }
    
    return $html;
}

/**
 * PHP 8.1 specific caching optimizations
 */
function firma_rehberi_php81_caching() {
    // Enable object caching
    if (!wp_using_ext_object_cache()) {
        add_action('init', 'firma_rehberi_enable_object_cache');
    }
    
    // Cache expensive operations
    add_action('init', 'firma_rehberi_cache_expensive_operations');
}

/**
 * Enable object cache
 */
function firma_rehberi_enable_object_cache() {
    // Set cache groups
    wp_cache_add_global_groups(array('firma_rehberi', 'company_data', 'seo_data'));
}

/**
 * Cache expensive operations
 */
function firma_rehberi_cache_expensive_operations() {
    // Cache schema data
    add_filter('firma_rehberi_company_schema', 'firma_rehberi_cache_schema_data', 10, 2);
    
    // Cache SEO data
    add_filter('firma_rehberi_seo_meta', 'firma_rehberi_cache_seo_data', 10, 2);
}

/**
 * Cache schema data
 */
function firma_rehberi_cache_schema_data($schema, $post_id) {
    $cache_key = 'schema_' . $post_id;
    $cached_schema = wp_cache_get($cache_key, 'firma_rehberi');
    
    if ($cached_schema === false) {
        wp_cache_set($cache_key, $schema, 'firma_rehberi', DAY_IN_SECONDS);
        return $schema;
    }
    
    return $cached_schema;
}

/**
 * Cache SEO data
 */
function firma_rehberi_cache_seo_data($meta, $post_id) {
    $cache_key = 'seo_' . $post_id;
    $cached_meta = wp_cache_get($cache_key, 'firma_rehberi');
    
    if ($cached_meta === false) {
        wp_cache_set($cache_key, $meta, 'firma_rehberi', DAY_IN_SECONDS);
        return $meta;
    }
    
    return $cached_meta;
}

/**
 * PHP 8.1 specific error handling
 */
function firma_rehberi_php81_error_handling() {
    // Set error reporting level
    if (defined('WP_DEBUG') && WP_DEBUG) {
        error_reporting(E_ALL & ~E_DEPRECATED & ~E_STRICT);
    } else {
        error_reporting(0);
    }
    
    // Custom error handler for theme
    set_error_handler('firma_rehberi_theme_error_handler', E_WARNING | E_NOTICE | E_USER_WARNING | E_USER_NOTICE);
}

/**
 * Theme error handler
 */
function firma_rehberi_theme_error_handler($errno, $errstr, $errfile, $errline) {
    // Only handle theme errors
    if (strpos($errfile, 'firma-rehberi-pro') === false) {
        return false;
    }
    
    // Log error
    error_log("Firma Rehberi Pro Error: $errstr in $errfile on line $errline");
    
    // Don't display in production
    if (!defined('WP_DEBUG') || !WP_DEBUG) {
        return true;
    }
    
    return false;
}

// Initialize PHP 8.1 optimizations
add_action('after_setup_theme', 'firma_rehberi_php81_caching');
add_action('init', 'firma_rehberi_php81_error_handling');
